package examples.dbkona;

import weblogic.db.jdbc.*;
import java.sql.*;
import java.util.Properties;

/**
 * This simple example shows how to connect and execute queries with
 * dbKona. You will need to have Oracle running the DEMO database,
 * with WebLogic jDriver for Oracle (or another Oracle JDBC driver). This example
 * is executed from the command line, and results are displayed to
 * standard out. For other examples of how to display query results,
 * check the server-side Java examples that use htmlKona and dbKona in
 * the examples/htmlkona and examples/servlets folders.
 * <p>
 * If you don't have the Oracle DEMO database, you can
 * use the file <tt><a href=../utils/ddl/demo.ddl>utils/ddl/demo.ddl</a></tt> to set it up.
 * <p>The following section describes how to build and run the examples:
 * <p>
 * <ol>
 * <li><a href="#Build the example">Build the example</a>
 * <li><a href="#Configure the server">Configure the server</a>
 * <li><a href="#Run the example">Run the example</a>
 * </ol>
 * <h3><a name="Build the example">Build the example</a></h3>
 * <ol>
 * <li>Set up your development shell as described in <a
 * href=../examples.html#environment>Setting up your environment for building and running the examples</a>.
 * <p>
 * <li>Compile the example by executing an <font face="Courier New" size = -1>ant</font> 
 * build script or a command.
 * <p>
 * <ul>
 * <li>An <font face="Courier New" size = -1>ant</font> build script is available in the
 * <font face="Courier New" size = -1>samples/examples/dbkona</font> directory. Enter the following command
 * to execute the build script:
 * <p>
 * <pre><font face="Courier New"><b>ant</b></font></pre>
 * <p>
 * <li>Enter the following command to compile the example:
 * <p>
 * <pre><font face="Courier New"><b>$ javac -d %CLIENT_CLASSES% query.java</b></font></pre>
 * </ol>
 * </ol>
 * <h3><a name="Configure the server">Configure the server</a></h3>
 * <ol>
 * <li><a href=../examples.html#startServer>Start the server with the examples configuration.</a>
 * <p><li>Bring up the Administration
 * Console in your browser with the URL:
 * <pre>http://<i>hostname</i>:<i>port</i>/console</pre>
 * where:
 * <dl>
 * <dt><i><font size=-1>hostname</font></i>
 * <dd>Host name of the WebLogic Server
 * <dt><i><font size=-1>port</font></i>
 * <dd>Port where the WebLogic Server is listening for connections
 * </dl>
 * <li>Click to expand the Services node in the left-hand pane.
 * <li>Click to expand the JDBC node in the left-hand pane.
 * <li>Click to expand the Connection Pools node in the left-hand pane.
 * <li>Select the oraclePool node.
 * <li>Select the Targets tab in the right-hand pane to display the Available and Chosen targets.
 * <li>Select the examplesServer target in the Available column.
 * <li>Click the mover control to move the examplesServer target to the Chosen column.
 * <li>Click the Apply button in the lower right-hand corner to save your assignment in the console configuration file.
 * </ol>
 * <h3><a name="Run the example"></a>Run the example</h3>
 * <ol>
 * <li>Run this example by executing the following command in your development shell:
 * <p>
 * <pre><font face="Courier New"><b>$ java examples.dbkona.query</b></font></pre>
 * </ol>
 *
 * @author        Copyright (c) 1996-2001 by BEA Systems, Inc. All Rights Reserved.
 */

public class query {

  /**
   * Here the JDBC connection is set up and created. A JDBC
   * Statement is executed, and its resulting data is used to
   * construct a dbKona QueryDataSet from the "emp" table
   * in the Oracle DEMO database.
   * <p>
   * A QueryDataSet simplifies the client-side management of
   * JDBC results by providing the infrastructure of a memory
   * cache. QueryDataSets also allow records to be fetched
   * incrementally. dbKona also provides methods associated
   * with DataSets to automatically generate SQL.
   * A QueryDataSet can be constructed with or without
   * a JDBC ResultSet, which is shown in this method.
   * <p>
   * After working with the DataSet, we close it, as well as
   * the JDBC ResultSet and the Statement used to execute the
   * query.
   */
  public static void main(String argv[]) {

    Connection conn = null;
    Statement stmt = null;
    ResultSet rs = null;

    // A QueryDataSet simplifies the client-side management of result
    // sets by providing an the infrastructure of a memory cache.  A
    // QueryDataSet can be constructed with or without a JDBC
    // ResultSet, as shown below.
    QueryDataSet qs = null;

    try {
      // For more information on making a connection, check the
      // dbKona Developers Guide.
      Properties props = new Properties();
      props.put("user",     "scott");
      props.put("password", "tiger");
      props.put("server",   "DEMO");

      Driver myDriver = (Driver) Class.forName("weblogic.jdbc.oci.Driver").newInstance();
      conn = myDriver.connect("jdbc:weblogic:oracle", props);

      // This a QueryDataSet can be constructed directly from a Connection
      qs = new QueryDataSet(conn, "select * from emp");
      qs.fetchRecords();
      System.out.println("Record count = " + qs.size());

      // Close the DataSet.
      qs.close();

      // QueryDataSets allow records to be fetched incrementally.
      qs = new QueryDataSet(conn, "select * from emp");
      while (!qs.allRecordsRetrieved()) {
	// Fetch 2 records at a time . . .
	qs.fetchRecords(2);
	System.out.println("Record count = " + qs.size());

	// Clear the records in the cache.
	qs.clearRecords();
      }

      // Close the DataSet.
      qs.close();

      stmt = conn.createStatement();
      stmt.execute("select * from emp");
      rs = stmt.getResultSet();

      // This QueryDataSet is constructed with a JDBC ResultSet.
      qs = new QueryDataSet(rs);
      qs.fetchRecords();

      System.out.println(qs.schema());

      for (int i = 0; i < qs.size(); i++) {
	Record rec = qs.getRecord(i);
	System.out.println(rec.getValue("empno").asString() + " - " +
			   rec.getValue("ename").asString()  + " - " +
			   rec.getValue("hiredate").asString());
      }

    } catch(Exception e) {
      System.err.println("Trouble while executing example");
      e.printStackTrace();
    } finally {
      // You should always close DataSets, ResultSets, and Statements
      // when you have finished working with them.
      try {
	qs.close();
      } catch (Exception e) {}

      try {
	rs.close();
      } catch (Exception e) {}

      try {
	stmt.close();
      } catch (Exception e) {}

      // Finally, close the connection.
      try {
	conn.close();
      } catch(Exception e) {}
    }
  }
}
