package examples.ejb20.cascadeDelete.one2many;

import java.rmi.RemoteException;

import java.sql.Date;
import java.util.Iterator;

import javax.ejb.DuplicateKeyException;
import javax.ejb.FinderException;
import javax.ejb.ObjectNotFoundException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.ejb.EJBException;
import javax.ejb.CreateException;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import javax.rmi.PortableRemoteObject;

public class BankBean implements SessionBean {

  final static boolean VERBOSE = true;
  final static int numBeans = 5;

  private SessionContext ctx;

  private AccountHome accountHome;
  private CustomerHome customerHome;


  public void createAccountAndCustomer() throws RemoteException {
    log("\nCreate Accounts and customers:");

    String accountId = "";
    String customerName = "";

    // Create accounts with varying balances and some null accountTypes.
    for (int i=0; i<numBeans; i++) {
      try {
        accountId = "account" + i;
        if (i % 2 == 0) {
          // Create a few of beans with a null accountType
          createAccount(accountId, i * 1000, null);
        } else {
          createAccount(accountId, i * 1000, "Savings");
        }
      } catch(CreateException ce) {
        log("Error creating account: " + ce.getMessage());
        throw new EJBException(ce);
      }

      try {
        customerName = "customer" + i;
        java.sql.Date lastDate = new Date(System.currentTimeMillis());
        createCustomer(customerName, i, new Integer(i), lastDate);
      } catch(DuplicateKeyException dke) {
        log("A customer with name " + accountId + " already exists!");
        throw new EJBException(dke);
      } catch(CreateException ce) {
        log("Error creating customer: " + ce.getMessage());
        throw new EJBException(ce);
      }
    }
    log("\nCreated " + numBeans + " Customers and " + numBeans + " Accounts");
  }

  public void establishRelationship() throws RemoteException {
    log("\nEstablish One-to-Many Relationships between Customer and Accounts:");

    String accountId = "";
    String customerName = "";

    Account acct = null;
    Customer cust = null;

    for (int i=0; i<numBeans; i++) {
      try {
        if (i % 3 == 0) {
          customerName = "customer" + i;
          cust = customerHome.findByPrimaryKey(customerName);
        }

        accountId = "account" + i;
        acct = accountHome.findByPrimaryKey(accountId);

        // add all accounts to customer_0
        log("Add account "+ acct.getAccountId() + " to " +
          cust.getName());
        cust.addAccount(acct);
        // add customer to all account from ID 0 to ID 4
        log("Add customer "+ cust.getName() + " to account " +
          acct.getAccountId());
          acct.setCustomer(cust);
      } catch(FinderException fe) {
        log("Error setting account: " + fe.getMessage());
        throw new EJBException(fe);
      } catch(Exception e) {
        log("Error setting account: " + e.getMessage());
        throw new EJBException(e);
      }
    }
  }

  public void printCustomers() throws RemoteException {
    log("\nPrint Customers:");

    String customerName = "";
    Customer cust = null;

    // print out the customers
    for (int i=0; i<numBeans; i++) {
      try {
        customerName = "customer" + i;
        cust = customerHome.findByPrimaryKey(customerName);

        log("Customer: " + cust.getName() + " has accounts:");
        Iterator iter = cust.getAllAccounts().iterator();
        while (iter.hasNext()) {
          Account acct = (Account)iter.next();
          log("\t  " + acct.getPrimaryKey() + " with balance " +
          acct.balance());
        }
      } catch(FinderException fe) {
        log("Customer: " + customerName + " not found");
      }
    }
  }

  public void printAccounts() throws RemoteException {
    log("\nPrint Accounts:");

    String accountId = "";
    Account acct = null;

    // print out the accounts
    for (int i=0; i<numBeans; i++) {
      try {
        accountId = "account" + i;
        acct = accountHome.findByPrimaryKey(accountId);

        log("Account: " + acct.getAccountId() + " has customers:");
        log("\t " + acct.getCustomer().getPrimaryKey());
      } catch(FinderException fe) {
        log("Account: " + accountId + " not found");
      }
    }
  }

  public void removeCustomer_CascadeDeleteAccounts(int i) throws RemoteException {
    log("\nCascade Delete:");
    log("Removing customer " + i + " and cascade delete its accounts");

    // Remove customer
    String customerName = "customer" + i;

    try {
      Customer cust = customerHome.findByPrimaryKey(customerName);
      cust.remove();
      System.out.println(customerName + " removed");
    } catch(Exception e) {
      log("Error removing customer0: " + e.getMessage());
      throw new EJBException(e);
    }
  }

  public void cleanup() throws RemoteException {
    log("\ncleanup...");

    String accountId = "";
    String customerName = "";

    Account acct = null;
    Customer cust = null;

    // cleanup
    for (int i=0; i<numBeans; i++) {
      try {
        customerName = "customer" + i;
        cust = customerHome.findByPrimaryKey(customerName);
        cust.remove();
        System.out.println(customerName + " removed");
      } catch(FinderException e) {
        // ignored
      } catch(Exception e) {
        log("Error during cleanup: " + e.getMessage());
        throw new EJBException(e);
      }
    }

    for (int i=0; i<numBeans; i++) {
      try {
        accountId = "account" + i;
        acct = accountHome.findByPrimaryKey(accountId);
        acct.remove();
        System.out.println(accountId + " removed");
      } catch(FinderException e) {
        // ignored
      } catch(Exception e) {
        log("Error during cleanup: " + e.getMessage());
        throw new EJBException(e);
      }
    }

    log("Done cleanup...\n\n");
  }


  /**
   * Create a new account with the given id and balance
   */
  private Account createAccount(String id, double balance, String accountType)
    throws CreateException, RemoteException
  {
    log ("Creating account " + id + " with a balance of " +
      balance + " account type " + accountType + "...");

    Account ac = (Account) PortableRemoteObject.narrow(
      accountHome.create(id, balance, accountType),
      Account.class);

    log("Account " + id + " successfully created");

    return ac;
  }

  /**
   * Create a new customer with the given name
   */
  private Customer createCustomer(String name, int age, Integer level, Date last)
    throws CreateException, RemoteException
  {
    log ("Creating customer " + name + "...");

    Customer cust = (Customer) PortableRemoteObject.narrow(
      customerHome.create(name, age, level, last), Customer.class);

   log("Customer " + name + " successfully created");

   return cust;
  }

  private static void log(String s) {
    if (VERBOSE) System.out.println(s);
  }


  //
  // EJB lifecycle methods
  //
  public void setSessionContext(SessionContext c) {

    ctx = c;

    try {
      Context ic    = new InitialContext();

      accountHome  = (AccountHome)  ic.lookup("one2many.AccountHome");
      customerHome = (CustomerHome) ic.lookup("one2many.CustomerHome");

    } catch (NamingException ne) {
      throw new EJBException(ne);
    }
  }



  public void ejbCreate() {}

  public void ejbActivate() {}

  public void ejbPassivate() {}

  public void ejbRemove() {}


}











