package examples.ejb20.cascadeDelete.one2many;

import java.rmi.RemoteException;
import java.sql.Date;
import java.util.Collection;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Hashtable;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.FinderException;
import javax.ejb.ObjectNotFoundException;
import javax.ejb.RemoveException;
import javax.ejb.EJBObject;
import javax.ejb.EJBException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;
import javax.transaction.UserTransaction;;

/**
 * This class demonstrates CMP relationships and EJB-QL by performing the following:
 * <p>
 * <ul><li>Create 5 seperate accounts and customers, with different initial balances and type.
 * <li>Establish relationships between beans.
 * <li>Prints out relationships.
 * <li>Remove customer[0].
 * <li>Prints out relationships after the removal.
 * </ul>
 * <p>Finally, remove all newly created accounts.
 * <p>
 * This class also illustrates how to lookup an EJB home in the JNDI tree.
 *
 * @author Copyright (c) 1998-2000 by BEA Systems, Inc. All Rights Reserved.
 */
public class Client {

  private String url;
  private BankHome bankHome;

  public Client(String url)
    throws NamingException {

    this.url       = url;

    bankHome = lookupBankHome();
  }

  /**
   * Runs this example from the command line. Example:
   * <p>
   * <tt>java examples.ejb20.cascadeDelete.one2many.Client "t3://localhost:7001"</tt>
   * <p>
   * The parameters are optional, but if any are supplied,
   * they are interpreted in this order:
   * <p>
   * @param url               URL such as "t3://localhost:7001" of Server
   */
  public static void main(String[] args) {
    System.out.println("\nBeginning examples.ejb20.one2many.Client...\n");

    String url       = "t3://localhost:7001";

    if (args.length == 1) {
      url = args[0];
    } else if (args.length > 1) {
      System.out.println("Usage: examples.ejb20.one2many.Client WebLogicURL");
      return;
    }

    Client client = null;

    try {
      client = new Client(url);
    } catch (NamingException ne) {
      log("Unable to look up the beans home: " + ne.getMessage());
      System.exit(1);
    }

    try {
      client.runExample();
    } catch (Exception e) {
      log("There was an exception while creating and using the Accounts.");
      log("This indicates that there was a problem communicating with " +
          "the server: "+e);
    }

    System.out.println("\nEnd examples.ejb20.one2many.Client...\n");
  }

  /**
   * Create 5 new accounts and customers, with different initial balances and type.
   * Establish one-to-many relationship between customer and accounts.
   * Find all the customers by name and importance.
   * When finished, remove the new accounts.
   */
  public void runExample() throws RemoteException
  {
    Bank bank = null;
    try {
      bank = createBank();
    } catch(CreateException ce) {
      log("Error creating bank: " + ce.getMessage());
      ce.printStackTrace();
      System.exit(1);
    }

    log("\nCreate Accounts and customers:");
    bank.createAccountAndCustomer();
    log("\nEstablish One-to-Many Relationships between Customer and Accounts:");
    bank.establishRelationship();

    log("\nPrint Customers and Accounts");
    bank.printCustomers();
    bank.printAccounts();

    int i = 0;
    log("\nCascade Delete:");
    log("Removing customer " + i + " and cascade delete its accounts");
    bank.removeCustomer_CascadeDeleteAccounts(i);
    log("\nPrint Customers and Accounts");
    bank.printCustomers();
    bank.printAccounts();

    log("\ncleanup...");
    bank.cleanup();
  }

  /**
   * Create a new bank
   */
  private Bank createBank()
    throws CreateException, RemoteException
  {
    log ("Creating bank ...");

    Bank bank = (Bank) PortableRemoteObject.narrow(
      bankHome.create(), Bank.class);

    log("Bank successfully created");

    return bank;
  }

  /**
   * Look up the account bean's home interface using JNDI.
   */
  private BankHome lookupBankHome()
    throws NamingException
  {
    Context ctx = getInitialContext();

    try {
      Object home = (BankHome) ctx.lookup("one2many.BankHome");
      return (BankHome) PortableRemoteObject.narrow(home, BankHome.class);

    } catch (NamingException ne) {
      log("The client was unable to lookup the EJBHome.  Please make sure " +
      "that you have deployed the ejb with the JNDI name " +
      "beanManaged.AccountHome on the WebLogic server at "+url);

      throw ne;
    }
  }

  /**
   * Get an initial context into the JNDI tree.
   *
   * Java2 clients can use the jndi.properties file to set the
   * INITIAL_CONTEXT_FACTORY and the PROVIDER_URL
   *  private Context getInitialContext() throws NamingException {
   *    return new InitialContext();
   *  }
   *
   *
   * Using a Properties object will work on JDK 1.1.x and Java2
   * clients
   */
  private Context getInitialContext() throws NamingException {
    try {
      // Get an InitialContext
      Properties h = new Properties();
      h.put(Context.INITIAL_CONTEXT_FACTORY,
          "weblogic.jndi.WLInitialContextFactory");
      h.put(Context.PROVIDER_URL, url);
      return new InitialContext(h);
    } catch (NamingException ne) {
      log("We were unable to get a connection to the WebLogic server at "+url);
      log("Please make sure that the server is running.");
      throw ne;
    }
  }

  private static void log(String s) {
    System.out.println(s);
  }
}







