package examples.i18n.logging;

import java.util.Locale;

import weblogic.i18n.Localizer;
import weblogic.i18ntools.L10nLookup;


/**
 * This example shows how to use the internationalized (I18n) logging interface
 * to access a simple message catalog. The methods and arguments are
 * defined in the I18nLog.xml catalog. This example uses the simple message catalog <tt>I18nSimple.xml</tt>.
 * 
 * <p>The following section describes how to build and run the examples:
 * <ol>
 * <li><a href="#Build the example">Build the example</a>
 * <li><a href="#Run the example">Run the example</a>
 * </ol>
 * <h3><a name="Build the example">Build the example</a></h3>
 * <ol>
 * <li>Set up your development shell as described in 
 * <a href=../../examples.html#environment>Setting up your environment</a>.
 * <p>We provide a build script for you to build the example, called <b><a href="build.xml">build.xml</a></b>.
 * <li>Run the build script using the following Java  ant command:
 * <br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; <font face="Courier New,Courier"><font size=-1>$
 * ant</font></font></li>
 * 
 * <p>The script builds the example and places the files in the
 * samples\examples\i18n\logging directory of your WebLogic Server installation.</ol>
 * <p>These scripts process the <tt>I18nLog.xml</tt> catalog, and produce the logging class
 * <tt>examples.i18n.logging.I18nLogLogger</tt>. This class contains static methods for logging messages to the
 * WebLogic Server error log.
 * </ol>
 * 
 * <h3><a name="Run the example">Run the example</a></h3>
 * <ol>
 * <li>Execute the following command in your development shell: 
 * <pre><b>$ java examples.i18n.logging.I18nLog</b></pre>
 * <p><li>After you finish reviewing the messages, change the locale of the
 * server's computer to <b>French</b> (in Windows, change the locale from the <b>Control Panel</b> -> <b>Regional Settings</b>) 
 * to see how the messages change that are written to the client log.
 * <p><li>Use the Message Editor to load the simple message catalog that you have built and configured, and display the log message catalog information.
 * <p><li>Start the Message Editor:
 * <pre><i>java </i>weblogic.il8ntools.gui.MessageEditor</pre>
 * <p><li>The basic Localizer provides access to the following catalog information:
 * <ul>
 * <li>Version
 * <li>L10n Package - package for catalog data
 * <li>I18n Package - package for Logger methods
 * <li>Subsystem - catalog subsystem
 * <li>Severity
 * <ul>
 * <li>debug (128)
 * <li>info (64)
 * <li>warning (32)
 * <li>error (8)
 * </ul>
 * <li>Message Body - the message text
 * <li>Stack option - whether or not to log a stack trace
 * </ul>
 * <p><li>By default, a passed throwable logs a stack trace along with the method. Messages can optionally be defined to not log a stack trace.
 * The logger methods return the message ID for applications that want to do more than just log these messages.
 * <p>The message ID is used to obtain the message attributes. The <tt>L10nLookup</tt> object provides access to the catalogs via Localizer classes. Localizers provide the access to individual messages.
 * Each log message catalog has two Localizers: 
 * <ul>
 * <li>General message information
 * <li>Detailed attributes
 * </ul>
 * <p>You will need to get the <tt>L10nLookup</tt> properties and use them to get the message Localizers.
 * The basic localizer <tt>lcl</tt> returns a default value of <tt>false</tt>, and the detail localizer <tt>lclDetail</tt> returns a default value of <tt>true</tt>.
 * </ol>
 * <p>In addition to the basic information described above, the detail Localizer provides additional information, including cause and action for the message.
 * 
 * Copyright (c) 2000-2001 by BEA Systems, Inc. All Rights Reserved.
 */


public  class I18nLog {

    public I18nLog() {}

    public static void main(String[] argv) {
      /**
       * This call just logs an info message. There are no arguments defined
       * for this method. 
       */
      I18nLogLogger.logEntry();
      /**
       * Here's an example of a message including a variety
       * of arguments.
       */
      I18nLogLogger.testArgs(I18nLog.class.getName(),argv.length);
      /**
       * If a Throwable is passed then it will result in a stack trace
       * being logged along with the method by default.
       */
      Throwable t = new Throwable("Test with stack trace");
      I18nLogLogger.logTrace(t);
      /** 
       * Messages can optionally be defined to not log a stack trace.
       */
      I18nLogLogger.logNoTrace(t);
      /**
       * The logger methods return the messageid for applications
       * that want to do more than just log these messages.
       */
      String messageId = I18nLogLogger.getId();
      I18nLogLogger.showId(messageId);
      /**
       * the message id can be used to obtain the different attributes
       * of a message. The L10nLookup object provides access to the catalogs
       * via Localizer classes. Localizers provide the access to individual
       * messages. Each log message catalog has two Localizers: one for
       * general message information and one for the detailed attributes.
       *
       * The basic Localizer provides access to catalog information:
       *     Version
       *     L10n Package - package for catalog data
       *     I18n Package - package for Logger methods
       *     Subsystem - catalog subsystem
       * For each message it also provides:
       *     Severity: debug (128), info (64), warning (32), error (8)
       *     Message Body - the message text
       *     Stack option - whether to log a stack trace
       *     
       * First get to the L10nLookup properties, then use them to get the
       * Localizer's for the message.
       */
      L10nLookup l10n = L10nLookup.getL10n(); 
      /**
       * This returns the basic Localizer (arg 3 = false)
       */
      Localizer lcl = l10n.getLocalizer(messageId,Locale.getDefault(),false);
      /**
       * This returns the detailed Localizer (arg 3 = true)
       */
      Localizer lclDetail = l10n.getLocalizer(messageId,Locale.getDefault(),true);
      /**
       * Use this appplications simple message catalog to display the
       * log message catalog information
       */
      I18nSimpleTextFormatter fmt = new I18nSimpleTextFormatter();
      System.out.println(fmt.version(messageId,lcl.getVersion()));
      System.out.println(fmt.l10nPackage(messageId,lcl.getL10nPackage()));
      System.out.println(fmt.i18nPackage(messageId,lcl.getI18nPackage()));
      System.out.println(fmt.subsystem(messageId,lcl.getSubSystem()));
      System.out.println(fmt.severity(messageId,lcl.getSeverity(messageId)));
      System.out.println(fmt.body(messageId,lcl.getBody(messageId)));
      System.out.println(fmt.stack(messageId,lcl.getStackTrace(messageId)));
      /**
       * Now for the detailed information.
       */
      System.out.println(fmt.detail(messageId,lclDetail.getDetail(messageId)));
      System.out.println(fmt.cause(messageId,lclDetail.getCause(messageId)));
      System.out.println(fmt.action(messageId,lclDetail.getAction(messageId)));

    }
}
