package examples.iiop.ejb.stateless.rmiclient;

import java.rmi.RemoteException;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.RemoveException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import examples.iiop.ejb.stateless.server.wls.*;

/**
 * This class illustrates calling a stateless Session Bean and performing
 * the following exercises:
 * <ul>
 * <li> Create a Trader
 * <li> Buy some shares using the Trader
 * <li> Sell some shares using the Trader
 * <li> Remove the Trader
 * </ul>
 *
 * @author Copyright (c) 1998 by WebLogic, Inc. All Rights Reserved.
 * @author Copyright (c) 1998-2001 by BEA Systems, Inc. All Rights Reserved.
 */

public class Client {

  private static final String JNDI_NAME = "TraderHome_iiop";

  private String url;
  private TraderHome home;
  private static boolean remote = false;
  
  public Client(String url)
    throws NamingException
  {

    this.url       = url;
    
    home = lookupHome();
  }

  /**
   * Runs this example from the command line. Example:
   * <p>
   * <tt>java examples.iiop.ejb.stateless.rmiclient.Client iiop://hostname:port <true|false></tt>
   * <p>
   * The parameters are optional, but if any are supplied,
   * they are interpreted in this order:
   * <p>
   * @param url               URL such as "iiop://localhost:7001" of Server
   */
  public static void main(String[] args) throws Exception {

    log("\nBeginning statelessSession.Client...\n");

    String url       = "iiop://localhost:7001";
    

    // Parse the argument list 
     if (args.length < 1 || args.length > 2) {
       System.out.println("Usage: java examples.iiop.ejb.stateless.rmiclient.Client iiop://hostname:port <true|false>");
      return;
    } else {
      url = args[0];
      if (args.length == 2
          && args[1].equals("true"))
        remote = true;
    }

    Client client = null;
    try {
      client = new Client(url);
    } catch (NamingException ne) {
      System.exit(1);
    }

    try {
      client.example();
    } catch (Exception e) {
      log("There was an exception while creating and using the Trader.");
      log("This indicates that there was a problem communicating with the server: "+e);
    } 

    log("\nEnd statelessSession.Client...\n");
  }

  /**
   * Runs this example.
   */
  public void example()
    throws CreateException, RemoteException, RemoveException
  {
    
    // create a Trader
    log("Creating a trader");
    Trader trader = (Trader) narrow(home.create(), Trader.class);

    String [] stocks = {"BEAS", "MSFT", "AMZN", "HWP" };

      // execute some buys
    for (int i=0; i<stocks.length; i++) {
      int shares = (i+1) * 100;
      log("Buying "+shares+" shares of "+stocks[i]+".");
      if (remote)
        trader.buyRemote(stocks[i], shares);
      else
        trader.buy(stocks[i], shares);
    }

    // execute some sells
    for (int i=0; i<stocks.length; i++) {
      int shares = (i+1) * 100;
      log("Selling "+shares+" shares of "+stocks[i]+".");
      if (remote)
        trader.sellRemote(stocks[i], shares);
      else
        trader.sell(stocks[i], shares);
    }

    // remove the Trader
    log("Removing the trader");
    trader.remove();

  }

  /**
   * RMI/IIOP clients should use this narrow function
   */
  private Object narrow(Object ref, Class c) {
    return PortableRemoteObject.narrow(ref, c);
  }

  /**
   * Lookup the EJBs home in the JNDI tree
   */
  private TraderHome lookupHome()
    throws NamingException
  {
    // Lookup the beans home using JNDI
    Context ctx = getInitialContext();
   
    try {
      Object home = ctx.lookup(JNDI_NAME);
      return (TraderHome) narrow(home, TraderHome.class);
    } catch (NamingException ne) {
      log("The client was unable to lookup the EJBHome.  Please make sure ");
      log("that you have deployed the ejb with the JNDI name "+JNDI_NAME+" on the WebLogic server at "+url);
      throw ne;
    }
  }

  /**
   * Using a Properties object will work on JDK130
   * clients
   */
  private Context getInitialContext() throws NamingException {
    
    try {
      // Get an InitialContext
      Properties h = new Properties();
      h.put(Context.INITIAL_CONTEXT_FACTORY,
        "com.sun.jndi.cosnaming.CNCtxFactory");
      h.put(Context.PROVIDER_URL, url);
      return new InitialContext(h);
    } catch (NamingException ne) {
      log("We were unable to get a connection to the WebLogic server at "+url);
      log("Please make sure that the server is running.");
      throw ne;
    }
  }

  /**
   * This is the Java2 version to get an InitialContext.
   * This version relies on the existence of a jndi.properties file in
   * the application's classpath.
   *
   */
//    private static Context getInitialContext()
//      throws NamingException
//    {
//      return new InitialContext();
//    }

  private static void log(String s) {
    System.out.println(s);
  }
  
}
