package examples.jdbc.mssqlserver4;

import java.io.*;
import java.sql.*;
import java.util.Properties;

/** This example shows how to create, call and remove a stored
 * procedure from a Microsoft SQL Server.
 *
 * * <p>
 * <h3>Build the Example</h3>  
 * <ol>
 * <li> Open a new command shell. 
 * <p><li>Set up this development shell as described in 
 * <a href=../../examples.html#environment>Setting up Your Environment for 
 * Building and Running the Examples</a>.
 * <p>
 * <li>Change your CLASSPATH if neccessary, depending on which version of WebLogic jDriver for Microsoft SQL Server
 * you are using. If you are using Version 7.0 you do not need to make changes. If you are using
 * Version 6.5, pre-pend your classpath as follows by adding the jar file: 
 * <pre>  $ <b>set CLASSPATH=%WL_HOME%/lib/mssqlserver4v65.jar;%CLASSPATH%</b></pre>
 * <p>
 * <li> Change the database URL, login, and password connection parameters defined in the
 * example source code to correspond to your Microsoft SQL Server configuration.
 * If you need more help, check the section on connecting
 * to a database in the programming guide, <a
 * href="http://e-docs.bea.com/wls/docs61/mssqlserver4/API_jmsq4.html">Using WebLogic jDriver for Microsoft SQL Server</a>.
 * <p>
 * <li>Compile this example using the following command line:
 * <pre>  $ <b>javac -d %CLIENT_CLASSES% ProcedureTests.java</b></pre>
 * 
 * </ol>
 * <p><h3>Run the Example</h3>
 * <ol>
 * <li>Execute the following command in your development shell: 
 * 
 * <pre><b>$ java examples.jdbc.mssqlserver4.ProcedureTests</b></pre>
 * 
 * </ol>
 * <h3>There's More</h3>
 *
 * For more information about the WebLogic jDriver for Microsoft SQL Server, see   <a
 * href="http://e-docs.bea.com/wls/docs61/mssqlserver4/API_jmsq4.html">Using WebLogic jDriver for Microsoft SQL Server</a>. 
 * <p>
 * 
 * @author      Copyright (c) 2000-2001 by BEA Systems, Inc. All Rights Reserved. 
 */
public class ProcedureTests
{
  public static void main(String [] argv)
  {
      String user = "sa";
      String password = "secret";
      String server = "myHost:1433";
      String url = "jdbc:weblogic:mssqlserver4";       

      try {
        for (int i = 0; i < argv.length; i++) 
        {
          if (argv[i].equals("-user")) {
            i++; 
            user = (argv[i].equals("null") ? "" : argv[i]);
          } 
          else if (argv[i].equals("-password")) {
            i++; 
            password = (argv[i].equals("null") ? "" : argv[i]);
          }
          else if (argv[i].equals("-server")) {
            i++; 
            server = (argv[i].equals("null") ? "" : argv[i]);
          }
          else if (argv[i].equals("-url")) {
            i++; 
            url = (argv[i].equals("null") ? "" : argv[i]);
          }
        }
      } catch(ArrayIndexOutOfBoundsException aiobe) {
        System.err.println("\nUsage: java examples.jdbc.mssqlserver4.ProcedureTests [options] \n\n" +
                           "where options include:\n" +
                           "    -user <user>            User name to be passed to database.\n" +
                           "    -password <password>    User password to be passed to database.\n" +
                           "    -server <server>        DNS name of database server.\n" +
                           "    -url <url>              URL of database.\n");
        System.exit(1);
      }

      // Change the username, password, and server to your database login account.  
      Properties props = new Properties();
      props.put("user", user);
      props.put("password", password);
      props.put("server", server);

      Driver myDriver = null;
      java.sql.Connection conn         = null;
      java.sql.Statement stmt          = null;
      java.sql.CallableStatement cstmt = null;

    try
      {
        String driverName = "weblogic.jdbc.mssqlserver4.Driver";
 
        System.out.println("\n\nStored procedure tests...\n");

        // Instantiate the jdbc driver then connect to the database
        myDriver = (Driver) Class.forName(driverName).newInstance();
        conn = myDriver.connect(url, props);   

        System.out.println("Connection established with " + url + " as " + conn + "\n");

        /*
          Create the stored procedure with this SQL text:

          procedure simpleProcedure(@par1 varchar(255) output, 
          @par2 varchar(255) output, @par3 varchar(255)) as
          select @par1 = @par1 + ' plus something'
          select @par2 = @par3 + ' plus something different'
          return 5 */
        try
          {
            String sql = "create procedure simpleProcedure"
              + "(@par1 varchar(255) output, @par2 varchar(255) output, @par3 varchar(255)) as\n"
              + "select @par1 = @par1 + ' plus something'\n"
              + "select @par2 = @par3 + ' plus something different'\n"
              + "return 5\n";

            System.out.println("Creating simpleProcedure. Here's the SQL:\n\n" + sql);

            stmt = conn.createStatement();
            stmt.execute(sql);
            System.out.println("Stored procedure was created.\n");
            stmt.close();
          } catch(SQLException sqle) {
              System.out.println("The procedure was not created because " + sqle);
          }

        try
          {
            // Call the procedure with a result and 3 parameters
            System.out.println("Calling simpleProcedure with result only...\n");

            cstmt = conn.prepareCall("{?= call simpleProcedure(?,?,?)}"); 

            cstmt.registerOutParameter(1,Types.INTEGER);     // Result is out parameter 1
            cstmt.registerOutParameter(2,Types.VARCHAR);     // Parameter 2 is in and out
            cstmt.setString(2,"Some text");                  // Set value for parameter 2
            cstmt.registerOutParameter(3,Types.VARCHAR);     // Parameter 3 is out only (no in value)
            
            cstmt.setString(4,"3.14159265358979312"); // Parameter 4 is
            // in only (let's use a number with plenty of digits)
            cstmt.execute();

            // Notice that before we can get the output parameters using
            // any of the getXXX methods we make sure all update counts
            // and/or results sets returned by the statement were read.
            // To do this we call dumpResults which will just skip through
            // them. We don't care about the actual results for this test.

            // Dump result counts and result sets (we're only testing
            // procedure parameters)
            dumpResults(cstmt); 
            
            System.out.println("Procedure result is: " + cstmt.getInt(1));
            System.out.println("Procedure @par1 is '" + cstmt.getString(2) + "'");
            System.out.println("Procedure @par2 is '" + cstmt.getString(3) + "'\n");
            cstmt.close();
          } catch(SQLException sqle) {
              System.out.println("Failed while calling the procedure because " + sqle);
          }

        try
          {
            // Call the procedure with a result and 3 parameters
            System.out.println("Calling simpleProcedure with 3 parameters and a result...\n");

            cstmt = conn.prepareCall("{?= call simpleProcedure(?,?,?)}"); 

            cstmt.registerOutParameter(1,Types.INTEGER);     // Result is out parameter 1
            cstmt.registerOutParameter(2,Types.VARCHAR);     // Parameter 2 is in and out
            cstmt.setString(2,"Some text");                  // Set value for parameter 2
            // Parameter 3 is out only (no in value)
            cstmt.registerOutParameter(3,Types.VARCHAR); 
            cstmt.setString(4,"3.14159265358979312");        // Parameter 4 is in only

            cstmt.execute();

            // Before we can get the output parameters using any of
            // the getXXX methods we make sure all update counts
            // and/or results sets returned by the statement were
            // read.  To do this we call dumpResults which just skips
            // through them. We don't care about the actual results.

            // Dump result counts and result sets.  We're only testing
            // procedure parameters.
            dumpResults(cstmt); 
            
            System.out.println("Procedure result is: " + cstmt.getInt(1));
            System.out.println("Procedure @par1 is '" + cstmt.getString(2) + "'");
            System.out.println("Procedure @par2 is '" + cstmt.getString(3) + "'\n");
            cstmt.close();
          } catch(SQLException sqle) {
              System.out.println("Failed while calling the procedure because " + sqle);
          }

        try
          {
            // Call the procedure with a result and 3 parameters. One
            // of the parameters comes from a stream as well.
            System.out.println("Calling simpleProcedure with 3 parameters and no result...\n");

            cstmt = conn.prepareCall("{call simpleProcedure(?,?,?)}"); 

            // Parameter 1 is both IN and OUT
            cstmt.registerOutParameter(1,Types.VARCHAR); 
            
            StringBufferInputStream stream = 
              new StringBufferInputStream("Some text coming from a stream");

            // Let's use a string stream for it
            cstmt.setAsciiStream(1,stream,stream.available()); 

            // Parameter 2 is an OUT parameter only. It returns @par3
            // + some string.
            cstmt.registerOutParameter(2,Types.VARCHAR); 

            // Parameter 3 is an IN parameter only (no OUT or return
            // value).
            cstmt.setString(3,"something"); 

            cstmt.execute();

            // Dump result counts and result sets (we're only testing procedure parameters)
            dumpResults(cstmt);                                      
            
            System.out.println("Procedure @par1 is '" + cstmt.getString(1) + "'");
            System.out.println("Procedure @par2 is '" + cstmt.getString(2) + "'\n");
            cstmt.close();
          } catch(SQLException sqle) {
              System.out.println("Failed while calling the procedure because " + sqle);
          }

        // Remove the stored procedure from the database.
        try
          {
            stmt = conn.createStatement(); 
            // This fails if procedure was not defined
            stmt.execute("drop procedure simpleProcedure");            
            stmt.close();
            System.out.println("Procedure was dropped from the database.\n");
          } catch(SQLException sqle) { }
      } catch(Exception e) {
          System.out.println("An exception was raised " + e);
      } finally { //close all connections, statements, etc in a finally block
          try { 
            if (conn != null)
              conn.close();
            if (stmt != null)
              stmt.close();
            if (cstmt != null)
              cstmt.close();
           } catch (SQLException sqle) {
               System.out.println("SQLException was thrown: " + sqle.getMessage());
           }
      }
  }

  /** 
   * Skips through result sets and/or update counts. 
   * @param statement, the executed statement.
   */
  static void dumpResults(Statement statement) throws SQLException
  {
    for(boolean hasMore = true ; hasMore ; )
      {
        ResultSet resultSet = statement.getResultSet();

        if(statement.getMoreResults() == false)
          {
            hasMore = (statement.getUpdateCount() != -1);
          }
      }
  }
}
