
package examples.jms.sender;

import java.io.*;
import javax.naming.*;
import javax.jms.*;
import javax.servlet.*;
import javax.servlet.http.*;

/**
 * This example shows how to send JMS messages from a Java servlet.
 * <p>The following sections describe how to:
 * <p>
 * <ol>
 * <li><a href="#Build the example">Build the example</a>
 * <li><a href="#Configure the server">Configure the server</a>
 * <li><a href="#Run the example">Run the example</a>
 * </ol>
 * <h3><a name="Build the example">Build the Example</a></h3>
 * <ol>
 * <li>
 * Set up your development shell, as described in 
 * <a href=../../examples.html#environment>Setting up 
 * your environment</a>.
 * <li>Compile the example by executing the following command or by executing the <a href=../../examples.html#buildScripts>build script</a> 
 * provided for this example in the <font face="Courier New" size = -1>samples/examples/jms/sender</font> 
 * directory. The script will perform the following step:
 * <p> 
 * Compile the SenderServlet as shown in this example for <b>Windows NT/2000</b>: 
 * <p><pre>  $<font face="Courier New"><b> javac -d %EX_WEBAPP_CLASSES% SenderServlet.java</b></font></pre>
 * </ol>
 * <h3><a name="Configure the server">Configure the Server</a></h3>
 * The following have been preset and/or preconfigured for your convenience:
 * <ul>
 * <li>
 * You need to choose one or more targets for the JMS server associated with
 * the topic. The examplesJMSServer server has been preconfigured to include
 * the examplesServer as a chosen target.
 * <li>
 * You need to deploy the jmssender servlet as part of a web application
 * hosted on the WebLogic Server. For your convenience, the jmssender 
 * servlet has been deployed on the examplesWebApp provided. 
 * For more information on deploying a servlet, refer to 
 * <a href="http://e-docs.bea.com/wls/docs61/servlet/admin.html">Administration and Configuration</a>
 * in <i>Programming WebLogic HTTP Servlets</i>. 
 * <li> 
 * In order to support persistent messaging, a
 * backing store must be configured and assigned to the JMS server. 
 * This example defines a JDBC backing store that uses a Cloudscape Java
 * database. An evaluation version of Cloudscape is included with WebLogic
 * Server and a demoPool database is provided.
 * </li></ul>
 * <p><b>Note:</b> If you have configured the server to run the 
 * <a href="../queue/package-summary.html"><tt>examples.jms.queue</tt></a> and 
 * <a href="../topic/package-summary.html"><tt>examples.jms.topic</tt></a>
 * examples, then no additional configuration changes are required to run
 * the <tt>SenderServlet</tt> example, 
 * and you can skip to the section
 * <a href="#Run the example">Run the Example</a>.
 * <p>To configure the server:
 * <ol>
 * <li>
 * <a href=../../examples.html#startServer>Start the server with the examples configuration</a>.
 * <li>
 * Bring up the <a href=../../examples.html#console>Administration Console</a> in your browser.
 * <li>
 * Click to expand the JMS node in the left pane.
 * <li>
 * Click to expand the JMS Connection Factories node in the left pane.
 * <li>
 * Choose the server target for the exampleQueue connection factory as follows:
 * <ol type="a">
 * <li>
 * Select the exampleQueue node. 
 * The exampleQueue connection factory configuration 
 * information displays in the right pane. 
 * <li>
 * Select the Targets tab in the right pane to display the Available and Chosen
 * targets.
 * <li>
 * Move the examplesServer target to the Chosen column and click the Apply button to 
 * save the assignment.
 * </ol>
 * <li>
 * Repeat step 5 to configure the exampleTopic connection factory, selecting 
 * the exampleTopic node instead of the exampleQueue node.
 * </ol>
 * <h3><a name="Run the example"></a>Run the Example</h3>
 * <ol>
 * <li>
 * <a href=../../examples.html#startServer>Start the server with the examples configuration</a> (if it not already started).
 * <li>
 * Load the servlet in a browser using the following URL:
 * <pre>  http://<i>hostname</i>:<i>port</i>/examplesWebApp/jmssender</pre>
 * where:
 * <dl>
 * <dt><i><font face="Courier New" size=-1>hostname</font></i>
 * <dd>Host name of the WebLogic Server.
 * <dt><i><font face="Courier New" size=-1>port</font></i>
 * <dd>Port where the WebLogic Server is listening for connections
 * (<font face="Courier New" size=-1>weblogic.system.ListenPort</font>). 
 * </dl>
 *
 * <p><li>
 * Enter a message to send and click the Send Message button. 
 * <li>
 * You can use the <a href="../queue/package-summary.html"><tt>QueueReceive</tt></a> and 
 * <a href="../topic/package-summary.html"><tt>TopicReceive</tt></a>
 * examples to observe messages being sent to the queue and topic, respectively,
 * based on the destination selected. For a queue, persistent messages will be 
 * sent even if the queue is not connected to the server and 
 * listening at the time the messages are sent.
 * </ol>
 *
 * @author Copyright (c) 1999-2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class SenderServlet extends HttpServlet
{
 /**
   * Defines the JMS connection factory.
   */
  public final static String JMS_FACTORY="weblogic.examples.jms.TopicConnectionFactory";
 /**
   * Defines the topic.
   */
  public final static String TOPIC="weblogic.examples.jms.exampleTopic";
 /**
   * Defines the queue.
   */
  public final static String QUEUE="weblogic.examples.jms.exampleQueue";

  /**
   * Processes HTTP requests.
   * @param req	HTTP servlet request
   * @param res	HTTP servlet response
   * @exception	IOException if problem with I/O operation
   * @exception ServletException if problem with servlet
   */
  public void service(HttpServletRequest req, HttpServletResponse res)
       throws IOException, ServletException
  {
    String literal;
    boolean persistent;
    boolean topicMsg;
    int priority;
    long ttl;
    String correlate;
    String replyto;
    String msgText="";
    try {
      if (req.getMethod().equals("GET")) {
        PrintWriter pw = new PrintWriter(res.getOutputStream(),true);
        printForm(pw);
      } else {
        topicMsg = 
          req.getParameterValues("dest")[0].equals("topic");
        persistent = 
          req.getParameterValues("persistent")[0].equals("persistent");
        priority =
          Integer.parseInt(req.getParameterValues("priority")[0]);
        ttl = 
          Long.parseLong(req.getParameterValues("timetolive")[0]);
        correlate = 
          req.getParameterValues("correlate")[0];
        replyto =
          req.getParameterValues("replyto")[0];
        msgText = 
          req.getParameterValues("msgtext")[0];

        if (topicMsg) {
          sendTopicMessage(persistent, priority, ttl, correlate, replyto, msgText);
        } else {
          sendQueueMessage(persistent, priority, ttl, correlate, replyto, msgText);
        }
      
        res.setContentType("text/html");
        res.setHeader("Pragma", "no-cache");
        PrintWriter pw = new PrintWriter(res.getOutputStream(),true);
        pw.println("<HTML><HEAD><TITLE>Message Status</TITLE></HEAD>");
        pw.println("<BODY BGCLOR=#FFFFFF><p><img src=images/BEA_Button_Final_web.gif align=right>");
	pw.println("<h1><FONT COLOR=#DB1260 FACE=\"Helvetica\">");
        pw.println("Message Status</FONT></h1>");
        pw.println("Message Submitted: "+msgText);
        pw.println("</BODY></HTML>");
        pw.close();
        return;
      }
    }
    catch (Exception e) {e.printStackTrace();
    }

  }
  /**
   * Displays the web page for message input.
   * @param pw	print writer
   * @exception Exception	if problem with printing form
   */

  public void printForm(PrintWriter pw) 
       throws Exception
  {
    pw.println("<html><head><title>JMS Message Submitter</title></head>");
    pw.println("<body BGCOLOR=#FFFFFF><FONT FACE=\"Helvetica\">");
    pw.println("<p><img src=images/BEA_Button_Final_web.gif align=right>");
    pw.println("<h1><font color=#DB1260 face=\"HELVETICA\">");
    pw.println("Submit a JMS Message</font></h1>");
    pw.println("<FORM METHOD=\"POST\" ACTION=\"./jmssender\">");
    pw.println("<TABLE bgcolor=#EEEEEE cellpadding=2 border=0 align=center>");
    pw.println("<TR><TD>Destination:</TD>");
    pw.println("<TD><select name=dest>");
    pw.println("<option value=topic selected>JMS Topic");
    pw.println("<option value=queue>JMS Queue");
    pw.println("</select></TR>");
    pw.println("<TR><TD>Message Type:</TD>");
    pw.println("<TD><select name=persistent>");
    pw.println("<option value=persistent selected>Persistent");
    pw.println("<option value=nonpersistent>Non Persistent");
    pw.println("</select></TR>");
    pw.println("<TR><TD>Priority:</td>");
    pw.println("<TD><select name=priority>");
    pw.println("<option value=0>0");
    pw.println("<option value=1>1");
    pw.println("<option value=2>2");
    pw.println("<option value=3>3");
    pw.println("<option value=4 selected>4");
    pw.println("<option value=5>5");
    pw.println("<option value=6>6");
    pw.println("<option value=7>7");
    pw.println("<option value=8>8");
    pw.println("<option value=9>9");
    pw.println("</select>");
    pw.println("</TR><TR><TD>Time to live:</td>");
    pw.println("<TD><select name=timetolive>");
    pw.println("<option value=0 selected>Never expires");
    pw.println("<option value=60000>one minute");
    pw.println("<option value=600000>ten minutes");
    pw.println("<option value=3600000>one hour");
    pw.println("<option value=86400000>one day");
    pw.println("</select>");
    pw.println("</TR>");
    pw.println("<TR><TD>Correlation ID:</TD><TD>");
    pw.println("<INPUT NAME=\"correlate\" SIZE=20></TD></TR>");
    pw.println("<TR><TD>Reply to:</TD><TD>");
    pw.println("<select name=replyto>");
    pw.println("<option value=none selected>none");
    pw.println("<option value=queue>ExampleQueue");
    pw.println("<option value=topic>ExampleTopic");
    pw.println("</select></TR>");
    pw.println("<TR><TD>Message Text:</TD><TD><INPUT NAME=\"msgtext\" SIZE=60></TD></TR>");
    pw.println("<TR><TD colspan=2 align=center>");
    pw.println("<INPUT TYPE=\"submit\" VALUE=\"Send Message\"></TD></TR>");
    pw.println("</TABLE></FORM></FONT></BODY></HTML>");
  }

  /**
   * Sends message to the topic.
   * @param persistent	persistency setting
   * @param priority	priority setting
   * @param ttl		time-to-live setting
   * @param correlate	correlation ID setting
   * @param replyto     reply-to setting
   * @param topicMessage	message
   * @exception NamingException if problem with JNDI context interface
   * @exception JMSException  if JMS fails send message due to internal error
   */
  public void sendTopicMessage(boolean persistent, int priority, 
                               long ttl, String correlate,
                               String replyto, String topicMessage)
       throws NamingException, JMSException
  {
    Context ctx = new InitialContext();
    TopicConnectionFactory tconFactory;
    TopicConnection connection;
    TopicSession session;
    TopicPublisher publisher;
    Topic topic;
    Queue queue;
    TextMessage msg;

    tconFactory = (TopicConnectionFactory) ctx.lookup(JMS_FACTORY);
    connection = tconFactory.createTopicConnection();
    session = connection.createTopicSession(false, Session.AUTO_ACKNOWLEDGE);
    topic = (Topic) ctx.lookup(TOPIC);
    publisher = session.createPublisher(topic);
    msg = session.createTextMessage();
    if (correlate.length() > 0)
      msg.setJMSCorrelationID(correlate);
    if (replyto.equals("topic")) {
      msg.setJMSReplyTo(topic);
    } 
    else if (replyto.equals("queue")) {
      try {
        queue = (Queue) ctx.lookup(QUEUE);
        msg.setJMSReplyTo(queue);
      } catch (NamingException ne) {
      }
    }
    msg.setText(topicMessage);  
    connection.start();

    publisher.publish(msg, 
                      persistent ? DeliveryMode.PERSISTENT
                                 : DeliveryMode.NON_PERSISTENT,
                      priority, 
                      ttl);

    publisher.close();
    session.close();
    connection.close();
  }

  /**
   * Sends a message to the queue. 
   * @param persistent	persistency setting
   * @param priority	priority setting
   * @param ttl		time-to-live setting
   * @param correlate	correlation ID setting
   * @param replyto     reply-to setting
   * @param topicMessage	message
   * @exception NamingException if problem with JNDI context interface
   * @exception JMSException  if JMS fails send message due to internal error
   */
  public void sendQueueMessage(boolean persistent, int priority, 
                               long ttl, String correlate,
                               String replyto, String topicMessage)
       throws NamingException, JMSException
  {
    Context ctx = new InitialContext();
    QueueConnectionFactory qconFactory;
    QueueConnection qcon;
    QueueSession qsession;
    QueueSender qsender;
    Queue queue;
    Topic topic;
    TextMessage msg;

    qconFactory = (QueueConnectionFactory) ctx.lookup(JMS_FACTORY);
    qcon = qconFactory.createQueueConnection();
    qsession = qcon.createQueueSession(false, Session.AUTO_ACKNOWLEDGE);
    queue = (Queue) ctx.lookup(QUEUE);
    qsender = qsession.createSender(queue);
    msg = qsession.createTextMessage();
    if (correlate.length() > 0)
      msg.setJMSCorrelationID(correlate);
    if (replyto.equals("queue")) {
      msg.setJMSReplyTo(queue);
    } 
    else if (replyto.equals("topic")) {
      topic = (Topic) ctx.lookup(TOPIC);
      msg.setJMSReplyTo(topic);
    }
    msg.setText(topicMessage);

    qcon.start();
    qsender.send(msg,
                 persistent ? DeliveryMode.PERSISTENT
                            : DeliveryMode.NON_PERSISTENT,
                 priority,
                 ttl);

    qsender.close();
    qsession.close();
    qcon.close();
  }

}

