package examples.jta.oracle;

import java.rmi.RemoteException;
import java.util.Collection;
import java.util.Properties;
import java.util.Vector;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.ObjectNotFoundException;
import javax.ejb.RemoveException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;
import javax.transaction.RollbackException;
import javax.transaction.UserTransaction;
import weblogic.transaction.Transaction;
import weblogic.transaction.TxHelper;


/**
 * This class demonstrates calling an entity EJBean.
 * <p>
 * <p>
 * This class also illustrates how to lookup an EJB home in the JNDI tree. 
 * 
 * @author Copyright (c) 1998 by WebLogic, Inc. All Rights Reserved.
 * @author Copyright (c) 1998-2000 by BEA Systems, Inc. All Rights Reserved.
 */

public class Client {

  private String url;
  
  private AccountHome home1;
  private AccountHome home2;

  public Client(String url)
    throws NamingException
  {

    this.url       = url;

    home1 = lookupAccountHome("examples-jta-oracle-SavingAccountHome");
    home2 = lookupAccountHome("examples-jta-oracle-CheckingAccountHome");
  }

  /**
   * Runs this example from the command line. Example:
   * <p>
   * <tt>java examples.jta.oracle.Client "t3://localhost:7001"</tt>
   * <p>
   * The parameters are optional, but if any are supplied,
   * they are interpreted in this order:
   * <p>
   * @param url               URL such as "t3://localhost:7001" of Server
   */
  public static void main(String[] args) throws NamingException {
    System.out.println("\nBeginning Client...\n");

    String url       = "t3://localhost:7001";
         
    // Parse the argument list 
     if (args.length != 1) {
      System.out.println("Usage: java examples.jta.oracle.Client t3://hostname:port");   
      return;
    } else {
      url = args[0];
    }

    
    Client client = null;

    try {
      client = new Client(url);
    } catch (NamingException ne) {
      log("Unable to look up the beans home: " + ne.getMessage());
      System.exit(1);
    }

    try {
      client.example();

    } catch (Exception e) {
      log("There was an exception while creating and using the Accounts.");
      log("This indicates that there was a problem communicating with the server: "+e);
    } 

    System.out.println("\nEnd Client...\n");
  }
     


  public void example() throws Exception
  {
    Account savingAcct;
    Account checkingAcct;

    savingAcct = findOrCreateAccount(home1, "ID: 1", 5000);
    checkingAcct = findOrCreateAccount(home2, "ID: 2", 1000);

    log("Savings Account: :"+savingAcct.getPrimaryKey()+
        " has a balance of "+savingAcct.balance());
    log("Checking Account: :"+checkingAcct.getPrimaryKey()+
        " has a balance of "+checkingAcct.balance());

    log("Transferring $100 from savings to checking account ...");
    
    UserTransaction utx = getUserTransaction();
    try {
      utx.begin();
    } catch (Exception ex) {
      log("Cannot create transaction");
      throw ex;
    }
    String txName = "[Funds Transfer]";
    Transaction tx = TxHelper.getTransaction();
    tx.setName(txName);
    log("TRANSACTION BEGUN: " + txName);
    log("Withdrawing $100 from Savings Account ...");
    savingAcct.withdraw(100);
    log("Depositing $100 to Checkings Account ...");
    checkingAcct.deposit(100);
    try {
      utx.commit();
    } catch (RollbackException ex) {
      log("Transaction rolled back already");
      throw ex;
    } 
    log("TRANSACTION COMMITTED: " + txName);

    log("Savings Account: :"+savingAcct.getPrimaryKey()+
        " has a balance of "+savingAcct.balance());
    log("Checking Account: :"+checkingAcct.getPrimaryKey()+
        " has a balance of "+checkingAcct.balance());

    log("Removing beans...");
    savingAcct.remove();
    checkingAcct.remove();
  }

  /**
   * If there already exists an account for this id, we will return
   * it.  Otherwise we will create a new account.
   */
  private Account findOrCreateAccount(AccountHome home, String id, double balance)
    throws CreateException, RemoteException, FinderException
  {
    try {
      log("Trying to find account with id: "+id);
      return (Account) PortableRemoteObject.narrow(home.findByPrimaryKey(id),
                                                   Account.class);
    } catch (ObjectNotFoundException onfe) {
      // the account id does not yet exist so create it.
      return (Account) PortableRemoteObject.narrow(home.create(id, balance),
                                                   Account.class);
    } 
  }

  /**
   * Look up the bean's home interface using JNDI.
   */
  private AccountHome lookupAccountHome(String homeName)
    throws NamingException
  {
    Context ctx = getInitialContext();
   
    try {
      Object home = ctx.lookup(homeName);
      return (AccountHome) PortableRemoteObject.narrow(home, AccountHome.class);

    } catch (NamingException ne) {
      log("The client was unable to lookup the EJBHome.  Please make sure " +
      "that you have deployed the ejb with the JNDI name " + homeName +
      " on the WebLogic server at "+url);

      throw ne;
    }
  }

  /**
   * Get an initial context into the JNDI tree.
   *
   * Java2 clients can use the jndi.properties file to set the
   * INITIAL_CONTEXT_FACTORY and the PROVIDER_URL
   *  private Context getInitialContext() throws NamingException {
   *    return new InitialContext();
   *  }
   *
   *
   * Using a Properties object will work on JDK 1.1.x and Java2
   * clients
   */
  private Context getInitialContext() throws NamingException {
    
    try {
      // Get an InitialContext
      Properties h = new Properties();
      h.put(Context.INITIAL_CONTEXT_FACTORY,
        "weblogic.jndi.WLInitialContextFactory");
      h.put(Context.PROVIDER_URL, url);
      return new InitialContext(h);
    } catch (NamingException ne) {
      log("We were unable to get a connection to the WebLogic server at "+url);
      log("Please make sure that the server is running.");
      throw ne;
    }
  }

  private UserTransaction getUserTransaction() throws NamingException {
    return (UserTransaction)getInitialContext().lookup("javax.transaction.UserTransaction");
  }

  private static void log(String s) {
    System.out.println(s);
  }
}
