/*
 * SampleClient.java
 *
 * Jaas sample client
 *
 * @author Copyright (c) 2000-2001 by BEA Systems, Inc. All Rights Reserved.
 */

package examples.security.jaas;

import java.io.*;
import java.util.*;
import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.TextOutputCallback;
import javax.security.auth.callback.TextInputCallback;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;
import javax.security.auth.login.FailedLoginException;
import javax.security.auth.login.AccountExpiredException;
import javax.security.auth.login.CredentialExpiredException;

/**
 * Sample client for JAAS user authentication
 */
public class SampleClient
{
  /**
  * Attempt to authenticate the user.
  */
  public static void main(String[] args)
  {

    if(args.length < 1)
    {
      System.out.println("Usage:   java examples.security.jaas.SampleClient URL [cert]");
      System.out.println("Example: java examples.security.jaas.SampleClient t3://localhost:7001");
      System.out.println("Example: java examples.security.jaas.SampleClient http://localhost:7001");
      System.out.println("Example: java examples.security.jaas.SampleClient t3s://localhost:7002 cert");
      System.exit(0);
    }

    LoginContext loginContext = null;

    /**
     * Set JAAS server url system property and create a LoginContext
     */
    try
    {
      // Set server url for SampleLoginModule
      Properties property = new Properties(System.getProperties());
      property.put("weblogic.security.jaas.ServerURL", args[0]);
      System.setProperties(property);

      // Set configuration class name to load SampleConfiguration
      property = new Properties(System.getProperties());
      property.put("weblogic.security.jaas.Configuration", "examples.security.jaas.SampleConfig");
      System.setProperties(property);

      // Set configuration file name to load sample configuration policy file
      property = new Properties(System.getProperties());
      property.put("weblogic.security.jaas.Policy", "Sample.policy");
      System.setProperties(property);

      // Create LoginContext, check if to use client cert authentication otherwise specify username/password
      if((args.length == 2) && (args[1].equals("cert")))
        loginContext = new LoginContext("CertSamplePolicy", new MyCallbackHandler());
      else
        loginContext = new LoginContext("SamplePolicy", new MyCallbackHandler());
    }
    catch(SecurityException se)
    {
      se.printStackTrace();
      System.exit(-1);
    }
    catch(LoginException le)
    {
      le.printStackTrace();
      System.exit(-1);
    }

    /**
     * Attempt authentication
     */
    try
    {
      // If we return without an exception, authentication succeeded
      loginContext.login();
    }
    catch(FailedLoginException fle)
    {
      System.out.println("Authentication Failed, " + fle.getMessage());
      System.exit(-1);
    }
    catch(AccountExpiredException aee)
    {
      System.out.println("Authentication Failed: Account Expired");
      System.exit(-1);
    }
    catch(CredentialExpiredException cee)
    {
      System.out.println("Authentication Failed: Credentials Expired");
      System.exit(-1);
    }
    catch(Exception e)
    {
      System.out.println("Authentication Failed: Unexpected Exception, " + e.getMessage());
      e.printStackTrace();
      System.exit(-1);
    }

    /**
     * Retrieve authenticated subject, perform SampleAction as Subject
     */
    Subject subject = loginContext.getSubject();
    SampleAction sampleAction = new SampleAction();
    Subject.doAs(subject, sampleAction);

    System.exit(0);
  }
}

/**
 * Implementation of the CallbackHandler Interface
 */
class MyCallbackHandler implements CallbackHandler
{
  public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException
  {
    for(int i = 0; i < callbacks.length; i++)
    {
      if(callbacks[i] instanceof TextOutputCallback)
      {
        // Display the message according to the specified type
        TextOutputCallback toc = (TextOutputCallback) callbacks[i];
        switch(toc.getMessageType())
        {
        case TextOutputCallback.INFORMATION:
          System.out.println(toc.getMessage());
          break;
        case TextOutputCallback.ERROR:
          System.out.println("ERROR: " + toc.getMessage());
          break;
        case TextOutputCallback.WARNING:
          System.out.println("WARNING: " + toc.getMessage());
          break;
        default:
          throw new IOException("Unsupported message type: " + toc.getMessageType());
        }
      }
      else if(callbacks[i] instanceof NameCallback)
      {
        // Prompt the user for the username
        NameCallback nc = (NameCallback) callbacks[i];
        System.err.print(nc.getPrompt());
        System.err.flush();
        nc.setName((new BufferedReader(new InputStreamReader(System.in))).readLine());
      }
      else if(callbacks[i] instanceof PasswordCallback)
      {
        // Prompt the user for the password
        PasswordCallback pc = (PasswordCallback) callbacks[i];
        System.err.print(pc.getPrompt());
        System.err.flush();

        // Note: JAAS specifies that the password is a char[] rather than a String
        String tmpPassword = (new BufferedReader(new InputStreamReader(System.in))).readLine();
        int passLen = tmpPassword.length();
        char[] password = new char[passLen];
        for(int passIdx = 0; passIdx < passLen; passIdx++)
          password[passIdx] = tmpPassword.charAt(passIdx);
        pc.setPassword(password);
      }
      else if(callbacks[i] instanceof TextInputCallback)
      {
        // Prompt the user for the username
        TextInputCallback callback = (TextInputCallback) callbacks[i];
        System.err.print(callback.getPrompt());
        System.err.flush();
        callback.setText((new BufferedReader(new InputStreamReader(System.in))).readLine());
      }
      else
      {
        throw new UnsupportedCallbackException(callbacks[i], "Unrecognized Callback");
      }
    }
  }
}


