/**
 * SampleLoginModule.java
 *
 * This LoginModule authenticates users with a username/password.
 * 
 * @author Copyright (c) 2000-2001 by BEA Systems, Inc. All Rights Reserved.
 */

package examples.security.jaas;

import java.util.Map;
import java.util.Hashtable;
import java.io.IOException;
import java.net.MalformedURLException;
import java.rmi.RemoteException;
import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.LoginException;
import javax.security.auth.login.FailedLoginException;
import javax.security.auth.spi.LoginModule;
import weblogic.security.auth.Authenticate;
import weblogic.jndi.Environment;

/**
 * Sample login module that performs password authentication.
 */


public class SampleLoginModule implements LoginModule 
{
  private Subject subject = null;
  private CallbackHandler callbackHandler = null;
  private Map sharedState = null;
  private Map options = null;
  private boolean debug = true;
  private String url = null;

  // Authentication status
  private boolean succeeded = false;
  private boolean commitSucceeded = false;

  // Username and password
  private String username = null;
  private String password = null;

  /**
   * Initialize
   */
  public void initialize(Subject subject,
                         CallbackHandler callbackHandler,
                         Map sharedState,
                         Map options) 
  {
    this.subject = subject;
    this.callbackHandler = callbackHandler;
    this.sharedState = sharedState;
    this.options = options;

    // Check options for login module debug enabled
    if(options != null)
    {
      Hashtable opt = (Hashtable) options;
      Object val = opt.get("debug");
      if((val != null) && ((String) val).equalsIgnoreCase("true"))
      {
        debug = true;
        System.out.println("SampleLoginModule.initialize(), debug enabled");
      }
    }

    try
    {
      // Retrieve WLS server URL string
      url = System.getProperty("weblogic.security.jaas.ServerURL");
      if(debug)
        System.out.println("SampleLoginModule.initialize(), URL " + url);
    }
    catch(NullPointerException npe)
    {
      System.err.println("Error: ServerURL Not Specified");
    }
    catch(IllegalArgumentException iae)
    {
      System.err.println("Error: ServerURL Not Specified");
    }
    catch(SecurityException se)
    {
      System.err.println("Error: Security Exception on accessing ServerURL Specification");
    }
  }

  /**
   * Authenticate the user by username and password passed in
   *
   * @return true in all cases 
   *
   * @exception FailedLoginException if the authentication fails. 
   *
   * @exception LoginException if this LoginModule
   *		is unable to perform the authentication.
   */
  public boolean login() throws LoginException 
  {
    // Verify that the client supplied a callback handler
    if(callbackHandler == null)
    {
      if(debug)
        System.out.println("SampleLoginModule.login(), no callback handler specifed");
      throw new LoginException("No CallbackHandler Specified");
    }

    // Populate callback list
    Callback[] callbacks = new Callback[2];
    callbacks[0] = new NameCallback("username: ");
    callbacks[1] = new PasswordCallback("password: ", false);
    try
    {
      // Prompt for username and password
      callbackHandler.handle(callbacks);

      // Retrieve username
      username = ((NameCallback) callbacks[0]).getName();
      if(debug)
        System.out.println("SampleLoginModule.login(), username " + username);

      // Retrieve password, converting from char[] to String
      char[] charPassword = ((PasswordCallback) callbacks[1]).getPassword();
      if(charPassword == null)
      {
        // Treat a NULL password as an empty password, not NULL
        charPassword = new char[0];
      }
      password = new String(charPassword);
      if(debug)
        System.out.println("SampleLoginModule.login(), password " + password);
    }
    catch(IOException ioe)
    {
      throw new LoginException(ioe.toString());
    }
    catch(UnsupportedCallbackException uce)
    {
      throw new LoginException("Error: Callback " + uce.getCallback().toString() + " Not Available");
    }

    // Populate weblogic environment and authenticate
    Environment env = new Environment();
    env.setProviderUrl(url);
    env.setSecurityPrincipal(username);
    env.setSecurityCredentials(password);
    try
    {
      // Authenticate user credentials, populating Subject
      Authenticate.authenticate(env, subject);
    }
    catch(RemoteException re)
    {
      System.err.println("Error: Remote Exception on authenticate, " + re.getMessage());
      throw new LoginException(re.toString());
    }
    catch(IOException ioe)
    {
      System.err.println("Error: IO Exception on authenticate, " + ioe.getMessage());
      throw new LoginException(ioe.toString());
    }
    catch(LoginException le)
    {
      System.err.println("Error: Login Exception on authenticate, " + le.getMessage());
      throw new LoginException(le.toString());
    }

    // Successfully authenticated subject with supplied info
    succeeded = true;
    return succeeded;
  }

  /**
   * This method is called if the LoginContext's overall authentication succeeded
   * (the relevant REQUIRED, REQUISITE, SUFFICIENT and OPTIONAL LoginModules succeeded).
   *
   * If this LoginModule's own authentication attempted failed, then this method removes
   * any state that was originally saved.
   *
   * @exception LoginException if the commit fails.
   *
   * @return true if this LoginModule's own login and commit
   *		attempts succeeded, or false otherwise.
   */
  public boolean commit() throws LoginException
  {
    if(succeeded)
    {
      commitSucceeded = true;
      return true;
    }
    else
    {
      username = null;
      password = null;
      return false;
    }
  }

  /**
   * This method is called if the LoginContext's
   * overall authentication failed.
   * If this LoginModule's own authentication attempt
   * succeeded (checked by retrieving the private state saved by the
   * login and commit methods),then this method cleans up any state that was originally saved.
   *
   * @exception LoginException if the abort fails.
   *
   * @return false if this LoginModule's own login and/or commit attempts
   *		failed, and true otherwise.
   */
  public boolean abort() throws LoginException
  {
    if(succeeded == false)
    {
      return false;
    }
    else if((succeeded == true) && (commitSucceeded == false))
    {
      // Login succeeded but overall authentication failed
      succeeded = false;
      username = null;
      password = null;
    }
    else
    {
      // Overall authentication succeeded and commit succeeded,
      // but a commit further down the authentication chain failed
      logout();
    }
    return true;
  }

  /**
   * Logout the user.
   *
   * @exception LoginException if the logout fails.
   *
   * @return true in all cases since this LoginModule
   *          should not be ignored.
   */
  public boolean logout() throws LoginException
  {
    succeeded = false;
    commitSucceeded = false;
    username = null;
    password = null;
    return true;
  }
}
