package examples.wlec.ejb.simpapp;

import javax.ejb.*;
import javax.naming.*;
import java.rmi.RemoteException;
import java.util.Hashtable;

/**
 * This class illustrates using a stateless SessionBean that calls 
 * an IIOP handler which in turn calls a CORBA Simpapp server. This class does the following:
 * <ul>
 * <li> Creates a converter, and performs conversion of strings.
 * <li> No persistence of state between calls to the SessionBean.
 * <li> Uses the EJBean to perform all calls to CORBA simpapp server.
 * <li> Searches the JNDI tree for an appropriate container.
 * </ul>
 * <p>
 * You'll need to add an IIOP pool to your <font face="Courier New" size=-1>weblogic.properties</font> file:
 * <pre># Demo IIOP pool
 *
 *   weblogic.CORBA.connectionPool.simplepool=\
 *     appaddrlist=//your host name:port # where the simpapp is running,\
 *     failoverlist=//your host name:port # where the simpapp is running,\
 *     minpoolsize=1,\
 *     maxpoolsize=3,\
 *     username=your user name,\
 *     userrole=role of the user,\
 *     domainname=simpapp
 * </pre>
 *
 * You'll need to deploy the EJB to your <font face="Courier New" size=-1>weblogic.properties</font> file:
 * <pre># Demo ejb deployment 
 *   weblogic.ejb.deploy=c:/weblogic/classes/wlec_ejb_simpapp.jar
 * </pre>
 *
 * @author Copyright (c) 1999-2001 by BEA Systems, Inc. All Rights Reserved.
 */

public class Client {
  static String url       = "t3://localhost:7001";
  static String user      = null;
  static String password  = null;

  /**
  * Runs this example from the command line. For example:
  * <p>
  * <pre><font face="Courier New">& <b>java examples.wlec.ejb.simpapp.Client <i>WebLogicURL user password</i></b></font></pre>
  * <p>
  * The parameters are optional, but if any are supplied,
  * they are interpreted in this order:
  * <p>
  * @param WebLogicURL       URL of the server such as <font face="Courier New" size=-1>"t3://localhost:7001"</font>
  * @param user              Username, default null
  * @param password          User password, default null. It should be at least 8 characters 
  *
  */

  public static void main(String[] args) {

    System.out.println("\nBeginning simpapp.Client...\n");

    // Parse the argument list 
    if ((args == null) || (args.length == 0)) {} 
    else
      for (int i = 0; i < args.length; i++) {
        switch(i) {
          case 0:
            url = args[i];
            break;
          case 1:
            user = args[i];
            break;
          case 2:
            password = args[i];
            break;
          default:
        }
      }

    try {
  
      // Create a converter object, which will later help us do conversion
      Context ctx = getInitialContext();
      ConverterHome converterH = (ConverterHome) ctx.lookup("simpapp.ConverterHome");
      Converter converter = converterH.create();
    
      String inData = "It Works";
      System.out.println("Start of Conversion " + "for: " + inData);
  
      try {  
        // To lower
        System.out.println("\nConverting to lower case: " + inData);
        ConverterResult res = converter.toLower(inData);
        System.out.println("\n...Converted: " + res.outData);

        // To Upper
        System.out.println("\nConverting to upper case: " + inData);
        res = converter.toUpper(inData);
        System.out.println("\n...Converted: " + res.outData);
      }
      
      catch (ProcessingErrorException pe) {
        System.out.println("Processing Error: " + pe);
      } 
      catch (Exception e) {
        System.out.println(":::::::::::::: Error :::::::::::::::::");
        e.printStackTrace();
      }

      if (converter != null) {
        try {
          // Remove converter
  
          System.out.println("\nRemoving Converter");
          converter.remove();
        }
        catch (Exception e) {
          System.out.println("Error removing converter ");
          e.printStackTrace();
        }
      }
    }
    catch (Exception e) {
      System.out.println(":::::::::::::: Error :::::::::::::::::");
      e.printStackTrace();
    }
    System.out.println("\nEnd simpapp.Client...\n");
  }
 
  /**
  * Gets an initial context.
  *
  * @return                  Context
  * @exception               java.lang.Exception if there is
  *                          an error in getting a context
  */
  static public Context getInitialContext() throws Exception {
    Hashtable h = new Hashtable();
    h.put(Context.INITIAL_CONTEXT_FACTORY,
        "weblogic.jndi.WLInitialContextFactory");
    h.put(Context.PROVIDER_URL, url);
    if (user != null) {
      System.out.println ("user: " + user);
      h.put(Context.SECURITY_PRINCIPAL, user);
      if (password == null) 
      password = "";
      h.put(Context.SECURITY_CREDENTIALS, password);
    } 
    return new InitialContext(h);
  }
}
