package examples.wtc.atmi.convsimp;

import java.rmi.RemoteException;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.RemoveException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import weblogic.wtc.jatmi.TPException;
import weblogic.wtc.jatmi.TPReplyException;
import weblogic.wtc.jatmi.TypedString;

/**
 * This class illustrates calling a stateful SessionBean and performing
 * the following exercises:
 * <ul>
 * <li> Create a TuxedoConversation
 * <li> Buy some shares using the TuxedoConversation
 * <li> Sell some shares using the TuxedoConversation
 * <li> Remove the TuxedoConversation
 * </ul>
 * 
 * @author Copyright (c) 1998 by WebLogic, Inc. All Rights Reserved.
 * @author Copyright (c) 1998-2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class Client {
  private static final String JNDI_NAME = 
    "tuxedo.services.TuxedoConversationHome";

  private int datasize = 1;
  private int iterations = 1;
  private String url;
  private String user;
  private String password;
  private TuxedoConversationHome home;

  public Client(int datasize, int iterations, String url, String user, 
                String password) throws NamingException {
    this.datasize = datasize;
    this.iterations = iterations;
    this.url = url;
    this.user = user;
    this.password = password;

    home = lookupHome();
  }

  /**
   * Runs this example from the command line. Example:
   * <p>
   * <tt>java weblogic.wtc.examples.simpapp.Client string-to-convert "t3://localhost:7001" scott tiger</tt>
   * <p>
   * All parameters except the first one are optional. The options must be
   * supplied in the order given below:
   * <p>
   * @param string-to-convert The string to have Tuxedo convert to upper case
   * @param url               URL such as "t3://localhost:7001" of Server
   * @param user              User name, default null
   * @param password          User password, default null
   */
  public static void main(String[] args) {
    log("\nBeginning statefulSession.Client...\n");

    String url = "t3://localhost:7001";
    String user = null;
    String password = null;
    int datasize = 1;
    int iterations = 1;

    // Parse the argument list
    switch (args.length) {

      case 5:
        password = args[4];

      // fall through
      case 4:
        user = args[3];

      // fall through
      case 3:
        url = args[2];

      // fall through
      case 2:
        try {
          iterations = Integer.parseInt(args[1]);
        } catch (NumberFormatException nfe) {
          iterations = 1;
        } 

      case 1:
        try {
          datasize = Integer.parseInt(args[0]);
        } catch (NumberFormatException nfe) {
          datasize = 1;
        } 

        break;

      case 0:
        System.out.println("Client [datasize] [iterations] [url] [user] [password]");
        System.exit(1);
    }

    Client client = null;

    try {
      client = new Client(datasize, iterations, url, user, password);
    } catch (NamingException ne) {
      System.exit(1);
    } 

    try {
      client.example();
    } catch (TPException te) {
      log("tperrno encountered: " + te);
    } catch (Exception e) {
      log("There was an exception while creating and using the TuxedoConversation.");
      log("This indicates that there was a problem communicating with the server: " 
          + e);
    } 

    log("\nEnd statefulSession.Client...\n");
  } 

  /**
   * Runs the example.
   */
  public void example() 
          throws CreateException, RemoteException, RemoveException, 
                 TPException {
    String converted = null;

    // Create a TuxedoConversation
    log("Creating TuxedoConversation\n");

    TuxedoConversation tconv = (TuxedoConversation) narrow(home.create(), 
            TuxedoConversation.class);

    log("datasize: " + this.datasize + " iterations: " + this.iterations);

    try {
      tconv.conversation(this.datasize, this.iterations);
    } catch (TPException te) {
      log("An error occured in the service.  tperrno:" + te);
    } 

    return;
  } 

  /**
   * RMI/IIOP clients should use this narrow function
   */
  private Object narrow(Object ref, Class c) {
    return PortableRemoteObject.narrow(ref, c);
  } 

  /**
   * Lookup the EJBs home in the JNDI tree
   */
  private TuxedoConversationHome lookupHome() throws NamingException {

    // Lookup the beans home using JNDI
    Context ctx = getInitialContext();

    try {
      Object home = ctx.lookup(JNDI_NAME);

      return (TuxedoConversationHome) narrow(home, 
                                             TuxedoConversationHome.class);
    } catch (NamingException ne) {
      log("The client was unable to lookup the EJBHome.  Please make sure ");
      log("that you have deployed the ejb with the JNDI name " + JNDI_NAME 
          + " on the WebLogic server at " + url);

      throw ne;
    } 
  } 

  /**
   * Using a Properties object will work on JDK 1.1.x and Java2
   * clients
   */
  private Context getInitialContext() throws NamingException {

    try {

      // Get an InitialContext
      Properties h = new Properties();

      h.put(Context.INITIAL_CONTEXT_FACTORY, 
            "weblogic.jndi.WLInitialContextFactory");
      h.put(Context.PROVIDER_URL, url);

      return new InitialContext(h);
    } catch (NamingException ne) {
      log("We were unable to get a connection to the WebLogic server at " 
          + url);
      log("Please make sure that the server is running.");

      throw ne;
    } 
  } 

  private static void log(String s) {
    System.out.println(s);
  } 

}

