package examples.wtc.atmi.qsample;

import java.rmi.RemoteException;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.RemoveException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import weblogic.wtc.jatmi.TPException;

/**
 * This class illustrates calling a stateful SessionBean and performing
 * the following exercises:
 * <ul>
 * <li> Create a Qsample
 * <li> Buy some shares using the Qsample
 * <li> Sell some shares using the Qsample
 * <li> Remove the Qsample
 * </ul>
 * 
 * @author Copyright (c) 1998 by WebLogic, Inc. All Rights Reserved.
 * @author Copyright (c) 1998-2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class Client {

  private static final String JNDI_NAME = "tuxedo.services.QsampleHome";

  private String to_convert;
  private String url;
  private String user;
  private String password;
  private QsampleHome home;

  public Client(String toConvert, String url, String user, 
                String password) throws NamingException {

    this.to_convert = toConvert;
    this.url = url;
    this.user = user;
    this.password = password;

    home = lookupHome();
  }

  /**
   * Runs this example from the command line. Example:
   * <p>
   * <tt>java examples.ejb20.basic.statefulSession.Client "t3://localhost:7001" scott tiger</tt>
   * <p>
   * The parameters are optional, but if any are supplied,
   * they are interpreted in this order:
   * <p>
   * @param url               URL such as "t3://localhost:7001" of Server
   * @param user              User name, default null
   * @param password          User password, default null
   */
  public static void main(String[] args) {
    log("\nBeginning statefulSession.Client...\n");

    String url = "t3://localhost:7001";
    String user = null;
    String password = null;
    String toConvert = null;

    // Parse the argument list
    switch (args.length) {

      case 4:
        password = args[3];

      // fall through
      case 3:
        user = args[2];

      // fall through
      case 2:
        url = args[1];

      // fall through
      case 1:
        toConvert = args[0];

        break;

      case 0:
        System.out.println("Client string-to-convert [url] [user] [password]");
        System.exit(1);
    }

    Client client = null;

    try {
      client = new Client(toConvert, url, user, password);
    } catch (NamingException ne) {
      System.exit(1);
    } 

    try {
      client.example();
    } catch (TPException te) {
      log("tperrno encountered: " + te);
    } catch (Exception e) {
      log("There was an exception while creating and using the Qsample.");
      log("This indicates that there was a problem communicating with the server: " 
          + e);
    } 

    log("\nEnd statefulSession.Client...\n");
  } 

  /**
   * Runs the example.
   */
  public void example() 
          throws CreateException, RemoteException, RemoveException, 
                 TPException {
    String converted = null;

    // Create a Qsample
    log("Creating trader\n");

    Qsample qsample = (Qsample) narrow(home.create(), Qsample.class);

    // Sell some stock
    log("converting " + this.to_convert);

    converted = qsample.qtoupper(this.to_convert);

    log("converted to: " + converted);
  } 

  /**
   * RMI/IIOP clients should use this narrow function
   */
  private Object narrow(Object ref, Class c) {
    return PortableRemoteObject.narrow(ref, c);
  } 

  /**
   * Lookup the EJBs home in the JNDI tree
   */
  private QsampleHome lookupHome() throws NamingException {

    // Lookup the beans home using JNDI
    Context ctx = getInitialContext();

    try {
      Object home = ctx.lookup(JNDI_NAME);

      return (QsampleHome) narrow(home, QsampleHome.class);
    } catch (NamingException ne) {
      log("The client was unable to lookup the EJBHome.  Please make sure ");
      log("that you have deployed the ejb with the JNDI name " + JNDI_NAME 
          + " on the WebLogic server at " + url);

      throw ne;
    } 
  } 

  /**
   * Using a Properties object will work on JDK 1.1.x and Java2
   * clients
   */
  private Context getInitialContext() throws NamingException {

    try {

      // Get an InitialContext
      Properties h = new Properties();

      h.put(Context.INITIAL_CONTEXT_FACTORY, 
            "weblogic.jndi.WLInitialContextFactory");
      h.put(Context.PROVIDER_URL, url);

      // if (user != null) {
      // log ("user: " + user);
      // h.put(Context.SECURITY_PRINCIPAL, user);
      // if (password == null)
      // password = "";
      // h.put(Context.SECURITY_CREDENTIALS, password);
      // }
      return new InitialContext(h);
    } catch (NamingException ne) {
      log("We were unable to get a connection to the WebLogic server at " 
          + url);
      log("Please make sure that the server is running.");

      throw ne;
    } 
  } 

  /**
   * This is the Java2 version to get an InitialContext.
   * This version relies on the existence of a jndi.properties file in
   * the application's classpath.
   * 
   */

  // private static Context getInitialContext()
  // throws NamingException
  // {
  // return new InitialContext();
  // }

  private static void log(String s) {
    System.out.println(s);
  } 

}

