package examples.dbkona;

import weblogic.db.jdbc.*;
import java.sql.*;
import java.util.Properties;

/**
 * This example constructs a dbKona TableDataSet for inserting, updating,
 * and deleting records from an Oracle database. To run this example,
 * you must have the Oracle DEMO database running. The connection to
 * the database is made with WebLogic jDriver for Oracle. This example is run
 * from the command line and results are displayed to standard
 * out.
 * <p>
 * For other examples of how to display query results, check the
 * server-side Java examples that use htmlKona and dbKona in the
 * examples/servlets directories.
 * <p>
 * If you don't have the Oracle DEMO database, you can
 * use the file <tt><a href=../utils/ddl/demo.ddl>utils/ddl/demo.ddl</a></tt> to set it up.
 *<p>The following section describes how to build and run the examples:
 * <p>
 * <ol>
 * <li><a href="#Build the example">Build the example</a>
 * <li><a href="#Configure the server">Configure the server</a>
 * <li><a href="#Run the example">Run the example</a>
 * </ol>
 * <h3><a name="Build the example">Build the example</a></h3>
 * <ol>
 * <li>Set up your development shell as described in <a
 * href=../examples.html#environment>Setting up your environment for building and running the examples</a>.
 * <p>
 * <li>Compile the example by executing an <font face="Courier New" size = -1>ant</font> 
 * build script or a command.
 * <p>
 * <ul>
 * <li>An <font face="Courier New" size = -1>ant</font> build script is available in the
 * <font face="Courier New" size = -1>samples/examples/dbkona</font> directory. Enter the following command
 * to execute the build script:
 * <p>
 * <pre><font face="Courier New"><b>ant</b></font>, or</pre>
 * <p>
 * <li>Enter the following command to compile the example:
 * <p>
 * <pre><font face="Courier New"><b>$ javac -d %CLIENT_CLASSES% records.java</b></font></pre>
 * </ol>
 * <h3><a name="Configure the server">Configure the server</a></h3>
 * <ol>
 * <li><a href=../examples.html#startServer>Start the server with the examples configuration.</a>
 * <p><li>Bring up the Administration
 * Console in your browser with the URL:
 * <pre>http://<i>hostname</i>:<i>port</i>/console</pre>
 * where:
 * <dl>
 * <dt><i><font size=-1>hostname</font></i>
 * <dd>Host name of the WebLogic Server
 * <dt><i><font size=-1>port</font></i>
 * <dd>Port where the WebLogic Server is listening for connections
 * </dl>
 * <li>Click to expand the Services node in the left-hand pane.
 * <li>Click to expand the JDBC node in the left-hand pane.
 * <li>Click to expand the Connection Pools node in the left-hand pane.
 * <li>Select the oraclePool node.
 * <li>Select the Targets tab in the right-hand pane to display the Available and Chosen targets.
 * <li>Select the examplesServer target in the Available column.
 * <li>Click the mover control to move the examplesServer target to the Chosen column.
 * <li>Click the Apply button in the lower right-hand corner to save your assignment in the console configuration file.
 * </ol>
 * <h3><a name="Run the example"></a>Run the example</h3>
 * <ol>
 * <li>Run this example by executing the following command in your development shell:
 * <p>
 * <pre><font face="Courier New"><b>$ java examples.dbkona.records</b></font></pre>
 * </ol>
 *
 * @author       Copyright (c) 1996-2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class records {

  /**
   * First the JDBC connection is set up and created. Using
   * the JDBC connection object, a TableDataSet is constructed
   * without using SQL -- by supplying the Connection object
   * and the name of the Oracle table. In this example, there
   * is code for inserting, updating, and deleting records
   * with SQL that is automatically generated.
   * <p>
   * Creating a TableDataSet with a KeyDef is also illustrated
   * in this sample code, which is required for
   * updates and deletes to the DBMS data. You can save
   * at either the record level or the table level with
   * dbKona.
   */
  public static void main(String argv[]) {

    Connection conn = null;
    TableDataSet ts1 = null;
    TableDataSet ts2 = null;

    try {
      // For more information on making a connection, check the
      // dbKona Developers Guide.
      Properties props = new Properties();
      props.put("user",     "scott");
      props.put("password", "tiger");
      props.put("server",   "DEMO");

      Driver myDriver = (Driver) Class.forName("weblogic.jdbc.oci.Driver").newInstance();
      conn = myDriver.connect("jdbc:weblogic:oracle", props);


      // A TableDataSet simplifies the task of inserting, updating, and
      // deleting records by automatically generating the DML SQL.

      // Here we insert 100 records.
      ts1 = new TableDataSet(conn, "emp");
      for (int i = 0; i < 100; i++) {
	Record rec = ts1.addRecord();
	rec.setValue("empno", i)
	  .setValue("ename",  "Person " + i)
	  .setValue("deptno",  i);
      }

      // Save new records. dbKona does selective saves, that is,
      // it saves only those records in the TableDataSet.
      System.out.println("Inserting " + ts1.size() + " records.");
      ts1.save();


      // Define a KeyDef for updates and deletes.
      KeyDef key = new KeyDef().addAttrib("empno");

      // Update the 100 records that we originally added.
      ts2 = new TableDataSet(conn, "emp", key);
      ts2.where("empno < 100");
      ts2.fetchRecords();

      for (int i = 0; i < ts2.size(); i++) {
	Record rec = ts2.getRecord(i);
	rec.setValue("ename", "Worker " + i);
      }

      // Save the updated records.
      System.out.println("Update " + ts2.size() + " records.");
      ts2.save();

    // Delete the same 100 records.
      ts2.reset();
      ts2.fetchRecords();

      for (int i = 0; i < ts2.size(); i++) {
	Record rec = ts2.getRecord(i);
	rec.markToBeDeleted();
      }

      // Delete records from server.
      System.out.println("Delete " + ts2.size() + " records.");
      ts2.save();
    } catch(Exception e) {
      System.err.println("Trouble while executing example");
      e.printStackTrace();
    } finally {
      // You should always close DataSets, ResultSets, and Statements
      // when you have finished working with them.
      try {
	ts1.close();
      } catch (Exception e) {}

      try {
	ts2.close();
      } catch (Exception e) {}

      // Finally, close the connections.
      try {
	conn.close();
      } catch(Exception e) {}
    }
  }
}
