package examples.ejb.extensions.readMostly;

import java.rmi.RemoteException;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

/**
 * This class illustrates how to use a read-only EJB and a read-write EJB
 * together to provide "read-mostly" access to a set of data.
 *
 * @author Copyright (c) 2000-2001 by BEA Systems, Inc. All Rights Reserved.
 */

public final class Client {

  private final static boolean verbose = true;

  private String url;
  private StockHome readerHome;
  private StockWriterHome writerHome;

  public Client(String url)
    throws NamingException
  {

    this.url       = url;

    readerHome = (StockHome)
      narrow(lookupHome("readMostly.StockReaderHome"), StockHome.class);
    writerHome = (StockWriterHome)
      narrow(lookupHome("readMostly.StockWriterHome"), StockWriterHome.class);
  }

  public static void main(String[] args) {
    System.out.println("\nBeginning readMostly.Client...\n");

    String url       = "t3://localhost:7001";
     
    // Parse the argument list 
     if (args.length != 1) {
      System.out.println("Usage: java examples.ejb.extensions.readMostly.Client t3://hostname:port");   
      return;
    } else {
      url = args[0];
    }
    
    Client client = null;

    try {
      client = new Client(url);
    } catch (NamingException ne) {
      System.exit(1);
    }

    try {
      client.example();
    } catch (Exception e) {
      log("There was an exception while creating and using the Beans.");
      log("This indicates that there was a problem communicating with the server: "+e);
    } 

    System.out.println("\nEnd readMostly.Client...\n");
  }

  public void example() throws CreateException, FinderException, RemoteException, RemoveException {

    String stock = "BEAS";
    log("Creating a StockWriter for " + stock);
    StockWriter writer = (StockWriter)
      narrow(writerHome.create(stock, 100, 1000), StockWriter.class);

    log("Looking up Stock info for " + stock);
    Stock reader = (Stock)
      narrow(readerHome.findByPrimaryKey(stock), Stock.class);
    log(reader.getSymbol() + ":");
    log("\tPrice: " + reader.getPrice());
    log("\tVolume: " + reader.getVolume());
    log("\t52 Week High: " + reader.get52weekHigh());
    log("\t52 Week Low: " + reader.get52weekLow());

    log("Removing the StockWriter");
    writer.remove();

  }

  /**
   *  RMI/IIOP clients should use PortableRemoteObject.narrow()
   */
  private Object narrow(Object ref, Class c) {
    return PortableRemoteObject.narrow(ref, c);
  }

  private Object lookupHome(String homeName)
    throws NamingException
  {
    // Lookup the beans home using JNDI
    Context ctx = getInitialContext();
   
    try {
      return ctx.lookup(homeName);
    } catch (NamingException ne) {
      log("The client was unable to lookup the EJBHome.  Please make sure ");
      log("that you have deployed the ejb with the JNDI name "+homeName+" on the WebLogic server at "+url);
      throw ne;
    }
  }

  /**
   * Java2 clients can use the jndi.properties file to set the
   * INITIAL_CONTEXT_FACTORY and the PROVIDER_URL
   */
//    private Context getInitialContext() throws NamingException {
//      return new InitialContext();
//    }

  /**
   * Using a Properties object will work on JDK 1.1.x and Java2
   * clients
   */
  private Context getInitialContext() throws NamingException {
    
    try {
      // Get an InitialContext
      Properties h = new Properties();
      h.put(Context.INITIAL_CONTEXT_FACTORY,
        "weblogic.jndi.WLInitialContextFactory");
      h.put(Context.PROVIDER_URL, url);
      return new InitialContext(h);
    } catch (NamingException ne) {
      log("We were unable to get a connection to the WebLogic server at "+url);
      log("Please make sure that the server is running.");
      throw ne;
    }
  }

  private static void log(String s) {
    System.out.println(s);
  }

}
