package examples.ejb.subclass;

import java.rmi.RemoteException;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.RemoveException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

/**
 * This class illustrates using an EJBean that subclasses from a
 * another bean.
 * <ul>
 * <li> It creates an instance of the ChildBean
 * <li> Calls a method that is overloaded by the ChildBean
 * <li> Calls a method that is inherited from the ParentBean
 * <li> Calls a method that's unique to the ChildBean
 * <li> Removes the ChildBean instance
 * </ul>
 *
 * @author Copyright (c) 1998 by WebLogic, Inc. All Rights Reserved.
 * @author Copyright (c) 1998-2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class Client {

  private static final String JNDI_NAME = "SubClass.ChildHome";

  private String url;
  
  private ChildHome home;

  public Client(String url)
    throws NamingException
  {

    this.url       = url;
    
    home = lookupHome();
  }

  /**
   * Runs this example from the command line. Example:
   * <p>
   * <tt>java examples.ejb.subclass.Client "t3://localhost:7001"</tt>
   * <p>
   * The parameters are optional, but if any are supplied,
   * they are interpreted in this order:
   * <p>
   * @param url               URL such as "t3://localhost:7001" of Server
   */
  public static void main(String[] args) {
    System.out.println("\nBegin subclass.Client...\n");

    String url       = "t3://localhost:7001";
         
    // Parse the argument list 
     if (args.length != 1) {
      System.out.println("Usage: java examples.ejb.subclass.Client t3://hostname:port");   
      return;
    } else {
      url = args[0];
    }

    Client client = null;

    try {
      client = new Client(url);
    } catch (NamingException ne) {
      System.exit(1);
    }

    try {
      client.example();
    } catch (Exception e) {
      log("There was an exception while creating and using the ChildBean.");
      log("This indicates that there was a problem communicating with the server: "+e);
    } 

    System.out.println("\nEnd subclass.Client...\n");
  }

  /**
   * Runs this example
   */
  public void example()
    throws CreateException, RemoteException, RemoveException
  {
    System.out.println("Creating ChildBean...\n");
    Child child = (Child) narrow(home.create(), Child.class);

    log("Calling method overloaded by ChildBean class");
    String message = child.sayHello();
    log(message);
    
    log("Calling method inherited by ChildBean class");
    message = child.nonOverloadedMethod();
    log(message);

    log("Calling method that only exists in ChildBean class");
    message = child.childOnlyMethod();
    log(message);
    
    System.out.println("Removing ChildBean...\n");
    child.remove();
  }



  /**
   * RMI/IIOP clients must use PortableRemoteObject.narrow()
   */
  private Object narrow(Object ref, Class c) {
    return PortableRemoteObject.narrow(ref, c);
  }


  private ChildHome lookupHome()
    throws NamingException
  {
    // Lookup the beans home using JNDI
    Context ctx = getInitialContext();
   
    try {
      Object home = ctx.lookup(JNDI_NAME);

      return (ChildHome) narrow(home, ChildHome.class);

    } catch (NamingException ne) {
      log("The client was unable to lookup the EJBHome.  Please make sure ");
      log("that you have deployed the ejb with the JNDI name "+JNDI_NAME+" on the WebLogic server at "+url);
      throw ne;
    }
  }

  /**
   * Java2 clients can use the jndi.properties file to set the
   * INITIAL_CONTEXT_FACTORY and the PROVIDER_URL
   */
//    private Context getInitialContext() throws NamingException {
//      return new InitialContext();
//    }

  /**
   * Using a Properties object will work on JDK 1.1.x and Java2
   * clients
   */
  private Context getInitialContext() throws NamingException {
    
    try {
      // Get an InitialContext
      Properties h = new Properties();
      h.put(Context.INITIAL_CONTEXT_FACTORY,
        "weblogic.jndi.WLInitialContextFactory");
      h.put(Context.PROVIDER_URL, url);
      return new InitialContext(h);
    } catch (NamingException ne) {
      log("We were unable to get a connection to the WebLogic server at "+url);
      log("Please make sure that the server is running.");
      throw ne;
    }
  }

  private static void log(String s) {
    System.out.println(s);
  }

}
