package examples.ejb20.sequence.mssqlserver;

import java.rmi.RemoteException;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Properties;
import java.util.Vector;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.FinderException;
import javax.ejb.ObjectNotFoundException;
import javax.ejb.RemoveException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

/**
 * This class illustrates how to set up a CMP EJBean to
 * automatically generate the primary key upon insert.
 * <p>
 * Create accounts and deposit some money into them.  The primary key will
 * be retrieved and printed after each creation.
 *
 * @author Copyright (c) 1998 by WebLogic, Inc. All Rights Reserved.
 * @author Copyright (c) 1998-2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class Client {

  private String url;
  private String accountId;
  private AccountHome home;

  public Client(String url)
    throws NamingException {

    this.url       = url;

    home = lookupHome();
  }

  /**
   * Runs this example from the command line. Example:
   * <p>
   * <tt>java examples.ejb20.sequence.mssqlserver.Client "t3://localhost:7001"</tt>
   * <p>
   * The parameter is optional.  If none is passed in, it t3://localhost:7001 will be used.
   * <p>
   * @param url               URL such as "t3://localhost:7001" of Server
   */
  public static void main(String[] args) {
    System.out.println("\nBeginning sequence.mssqlserver.Client...\n");

    String url       = "t3://localhost:7001";

    Client client = null;

    // Parse the argument list
    switch(args.length) {
      case 1:
        url       = args[0];
        break;
    }

    try {
      client = new Client(url);
    } catch (NamingException ne) {
      log("Unable to look up the beans home: " + ne.getMessage());
      System.exit(1);
    }

    try {
      client.runExample();
    } catch (Exception e) {
      log("There was an exception while creating and using the Accounts.");
      log("This indicates that there was a problem communicating with the server: "+e);
    }

    System.out.println("\nEnd sequence.mssqlserver.Client...\n");
  }

   /**
    * Deposit and print the primary key of the newly created Account.
    */
   public void runExample() {
     log("Starting example...\n");

     double initialBalance = 30000;
     String accountType = "Savings";
     Account ac = null;
     try {
       ac = createAccount(initialBalance, accountType);
       System.out.println("just created Account with auto-generated primary key " + ((Integer)ac.getPrimaryKey()).intValue());
     } catch(Exception e) {
       log("Error creating account: " + e.getMessage());
       System.exit(1);
     }

     initialBalance = 1000;
     accountType = "Checking";
     ac = null;
     try {
       ac = createAccount(initialBalance, accountType);
       System.out.println("just created Account with auto-generated primary key " + ((Integer)ac.getPrimaryKey()).intValue());
     } catch(Exception e) {
       log("Error creating account: " + e.getMessage());
       System.exit(1);
     }


     log("End example...\n");
   }

  /**
   * Create a new account with the given id and balance
   */
  private Account createAccount(double balance, String accountType)
    throws CreateException, RemoteException
  {
    log ("Creating account with a balance of " +
      balance + " account type " + accountType + "...");

    Account ac = (Account) PortableRemoteObject.narrow(
      home.create(balance, accountType), Account.class);

   log("Account successfully created");

   return ac;
  }

  /**
   * Look up the bean's home interface using JNDI.
   */
  private AccountHome lookupHome()
    throws NamingException
  {
    Context ctx = getInitialContext();

    try {
      Object home = (AccountHome) ctx.lookup("ejb20-sequence-mssqlserver-AccountHome");
      return (AccountHome) PortableRemoteObject.narrow(home, AccountHome.class);

    } catch (NamingException ne) {
      log("The client was unable to lookup the EJBHome.  Please make sure " +
      "that you have deployed the ejb with the JNDI name " +
      "ejb20-sequence-mssqlserver-AccountHome on the WebLogic server at "+url);

      throw ne;
    }
  }

  /**
   * Get an initial context into the JNDI tree.
   *
   * Java2 clients can use the jndi.properties file to set the
   * INITIAL_CONTEXT_FACTORY and the PROVIDER_URL
   *  private Context getInitialContext() throws NamingException {
   *    return new InitialContext();
   *  }
   *
   *
   * Using a Properties object will work on JDK 1.1.x and Java2
   * clients
   */
  private Context getInitialContext() throws NamingException {

    try {
      // Get an InitialContext
      Properties h = new Properties();
      h.put(Context.INITIAL_CONTEXT_FACTORY,
          "weblogic.jndi.WLInitialContextFactory");
      h.put(Context.PROVIDER_URL, url);
      return new InitialContext(h);
    } catch (NamingException ne) {
      log("We were unable to get a connection to the WebLogic server at "+url);
      log("Please make sure that the server is running.");
      throw ne;
    }
  }

  private static void log(String s) {
    System.out.println(s);
  }

}
