package examples.i18n.simple;

import java.util.Locale;
import java.text.MessageFormat;

import weblogic.i18n.Localizer;
import weblogic.i18ntools.L10nLookup;

/**
 * @author Copyright (c) 2000-2001 by BEA Systems, Inc. All Rights Reserved.
 */

/**
 * This example shows various ways of internationalizing an application
 * using simple message catalogs.
 * <p>
 * Usage: java examples.i18n.simple.HelloWorld [lang [country]]
 * <p>
 * lang is a 2 character ISO language code. e.g. "en"
 * country is a 2 character ISO country code. e.g. "US"
 * <p>
 * Usage of any of the languages supported by this example presumes
 * the existence of the appropriate OS localization software and character
 * encodings.
 * <p>
 * The example comes with catalogs for English (the default) and  French. 
 * The catalog source is in the following files, and were built 
 * using the catalog editing utility, weblogic.i18ntools.gui.MessageEditor.
 * <p>
 * <pre>
 * English(base language)       ../msgcat/Helloworld.xml
 * French                       ../msgcat/fr/FR/HelloWorld.xml
 * </pre>
 * <p>
 * To build this example run the build.sh(UNIX) or build.cmd (NT) scripts from
 * the examples/i18n/simple directory. CLIENT_CLASSES must be set up and
 * needs to be in the classpath when running the example.
 */


public final class HelloWorld {

  public static void main(String[] args) {
    /*
     * The easiest method for displaying localized text is to 
     * instantiate the generated formatter class for the HelloWorld catalog.
     * This class contains convenience methods that return localized text for
     * each message defined in the catalog.  The class name is
     * the catalog name followed by "TextFormatter". 
     *
     * Normally, one would use the default constructor to obtain 
     * formatting in the current locale. In this example we'll use a locale
     * based on  arguments to construct the TextFormatter. 
     */
    Locale lcl;
    if (args.length == 0) { // default is default locale for JVM
      lcl = Locale.getDefault();
    }
    else {
      // get the language code
      String lang = args[0];
      String country = "";
      if (args.length >= 2) {
        // get the country code
        country = args[1];
      }
      lcl = new Locale(lang,country);
    }
    /*
     * get formatter in appropriate locale
     */  
    HelloWorldTextFormatter fmt = new HelloWorldTextFormatter(lcl);
    /*
     * print the text in the current locale
     */
    System.out.println(fmt.helloWorld());
    /*
     * Alternatively,  text can be accessed and formatted manually. In this
     * case you must obtain the Localizer class for the catalog. The Localizer
     * class is  formed from the l10n_package attribute in the catalog, the
     * catalog name, and the string "TextLocalizer".
     */
    Localizer l10n = L10nLookup.getLocalizer
      (lcl,"examples.i18n.simple.HelloWorldTextLocalizer");
    System.out.println(l10n.get("HELLO_AGAIN"));
    /*
     * If the message accepts arguments, then they can just be passed to the
     * method defined for the message.
     */
    System.out.println(fmt.nthHello(3));
    /*
     * If using the manual method then you must manually apply the argument to
     * the text using the MessageFormat class.
     */
    String text = l10n.get("NTH_HELLO");
    Object[] argary = {new Integer(4)};
    System.out.println(MessageFormat.format(text,argary));
    /*
     * The Localizer class also provides methods for accessing catalog information.
     */
    System.out.println(fmt.version(l10n.getVersion()));
    System.out.println(fmt.l10nPackage(l10n.getL10nPackage()));
    System.out.println(fmt.i18nPackage(l10n.getI18nPackage()));
    System.out.println(fmt.subSystem(l10n.getSubSystem()));
  }
}
