package examples.io;

import java.util.*;
import java.io.*;
import weblogic.common.*;
import weblogic.io.common.*;
import javax.naming.*;
import weblogic.jndi.*;

/**
 * This example illustrates the use of WebLogic File Services.  
 * HelloWorld looks for a file on the server, writes "Hello World" to
 * the file if it is NOT there, reads the first 11 bytes of the file, 
 * and displays them.
 * <p>The following sections cover what to do:
 * <p><ol>
 * <li><a href="#Build the Example">Build the Example</a>
 * <li><a href="#Configure the Server">Configure the Server</a>
 * <li><a href="#Run the Example">Run the Example</a>
 * </ol>
 * <h3><a name="Build the Example">Build the Example</a></h3>
 * <ol>
 * <li>Set up your development shell, as described in 
 * <a href=../examples.html#environment>Setting up your environment for building and running the examples</a>. 
 * <p>
 * <li>Compile the example by executing the following command or by executing the <a href=../examples.html#buildScripts>build script</a> 
 * provided for this example in the <font face="Courier New" size = -1>samples/examples/io</font> 
 * directory. The script will perform the following step:
 * <p> 
 * <ol type=a>
 * <li>Compile the HelloWorld class as shown in this example for <b>Windows NT/2000</b>:
 * <p>
 * <pre><font face="Courier New">  <b>$ javac -d %CLIENT_CLASSES% HelloWorld.java</b></font></pre>
 * </ol>
 * </ol>
 * <h3><a name="Configure the Server">Configure the Server</a></h3>
 * <ol>
 * <li><a href=../examples.html#startServer>Start the server with the examples 
 * configuration</a> in a new command shell.
 * <li>Bring up the <a href=../examples.html#console>Administration Console</a> in your browser. 
 * <li>Click to expand the Services node in the left-hand pane.
 * <li>Select the FileT3 node in the left-hand pane. 
 * <li>Click on <b>Register a new File T3</b> link in the right-hand pane to display Configuration, Targets and Notes tab.
 * <li>Select the Configuration tab in the right-hand pane to display the Name and Path field.
 * <li>Register a fileSystem that the WebLogic Server maps to a real path that is valid on the server host. For example, to map the fileSystem "mySys" to "c:\temp" : 
 * <pre> Name: mySys</pre>
 * <pre> Path: c:\temp</pre>
 * <li>Click the Create button to save the assignment. 
 * <li>Select the Targets tab in the right-hand pane to display the Available and Chosen targets.
 * <li>Move the examplesServer from the Target column to the Chosen column.
 * <li>Click the Apply button to save the assignment.
 * </ol>
 * <h3><a name="Run the Example"></a>Run the Example</h3>
 * <ol>
 * <li>In your development shell, run the "HelloWorld" example from the command
 * line of the same machine as your WebLogic Server, listening on port
 * 7001, with a registered fileSystem called "mySys":
 *
 * <pre><font face="Courier New">  <b>$ java examples.io.HelloWorld t3://<i>hostname</i>:<i>port</i> mySys</b></font></pre>
 * where: 
 * <dl>
 * <dt><i><font size=-1>hostname</font></i>
 * <dd>Host name of the WebLogic Server.
 * <dt><i><font size=-1>port</font></i>
 * <dd>Port where the WebLogic Server is listening for connections.
 * </dl>
 * <p>
 * After compiling this directory, call HelloWorld with three arguments:
 * <ul>
 *     <li> The URL of the WebLogic Server
 *     <li> The name of the file system
 *     <li> The name of the target file
 * </ul>
 * 
 *<p> You may also call HelloWorld with two arguments:
 * 
 * <ul>
 * <li>URL of your WebLogic Server
 * <li>Name of the target file
 * </ul>
 * 
 * in which case it will use the empty file system name. The empty file
 * system name refers to the local file system on the client. File names
 * are always absolute on a remote file system, but may be relative on
 * a local file system.
 * </ol>  
 * 
 * @author Copyright (c) 1997-1998 by WebLogic, Inc. All Rights Reserved.
 * @author Copyright (c) 1999-2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class HelloWorld {
  
  /**
   * Runs the HelloWorld file example.
   */
  public static void main(String[] argv) {
    String url;
    String fileSystemName;
    String fileName;
    if (argv.length == 2) {
      url = argv[0];
      fileSystemName = "";  // Use the local file system on the client.
      fileName = argv[1];
    }
    else if (argv.length == 3) {
      url = argv[0];
      fileSystemName = argv[1];
      fileName = argv[2];
    }
    else {
      System.out.println("Usage:   java examples.io.HelloWorld WebLogicURL fileSystemName fileName");
      System.out.println("Example: java examples.io.HelloWorld t3://localhost:7001 users test");
      return;
    }
    
    Context ctx = null;
    try {
      // Look-up T3Services context on the WebLogic Server...
      Hashtable env = new Hashtable();
      env.put(Context.PROVIDER_URL, url);
      env.put(Context.INITIAL_CONTEXT_FACTORY, 
              weblogic.jndi.WLInitialContextFactory.class.getName());
      ctx = new InitialContext(env);
      T3ServicesDef t3s = (T3ServicesDef)ctx.lookup("weblogic.common.T3Services");
      
      // Get the file system and the file
      System.out.println("Getting the file system " + fileSystemName);
      T3FileSystem fileSystem = t3s.io().getFileSystem(fileSystemName);
      System.out.println("Getting the file " + fileName);
      T3File file = fileSystem.getFile(fileName); 
      
      if (file.exists()) {
	// The file exists. Don't do anything. 
	System.out.println("The file already exists");
      }
      else {
	// The file does not exist. Create it.
	byte[] b = new String("Hello World").getBytes();

        // Get an OutputStream and write to the file
        System.out.println("Writing to the file");
	OutputStream os = file.getFileOutputStream();
	os.write(b);
	os.close();
      }
      
      // Get an InputStream and read from the file
      byte b[] = new byte[11];
      System.out.println("Reading from the file");
      InputStream is = file.getFileInputStream();
      is.read(b);
      is.close();
      
      // Report the result
      String result = new String(b);
      System.out.println("File contents: " + result);      
    }
    
    catch (Exception e) {
      System.out.println("The following exception occurred while " +
                         "running the HelloWorld example.");
      e.printStackTrace();
      if (!fileSystemName.equals("")) {
	System.out.println("Make sure the WebLogic Server at " + url + " was started with");
	System.out.println("the property weblogic.io.fileSystem." + fileSystemName + " set.");
      }
    }
    finally {
      // Always disconnect from the WebLogic Server
      try {
	if (ctx != null) {ctx.close();}
      }
      catch (Exception e) {}
    }
  }
}







