package examples.jdbc.datasource;
import java.sql.*;
import java.util.*;
import javax.naming.*;

/**
 * This example demonstrates obtaining a database connection using a <b>DataSource</b> object.
 * <P>
 * You use the Administration Console as well as the command line to build, 
 * compile, and run this example. 
 *
 * <p><h3>Build the Example</h3>  
 * <ol>
 * <li> Open a new command shell. 
 * <p>
 * <li>Set up this development shell as described in 
 * <a href=../../examples.html#environment>Setting up Your Environment for 
 * Building and Running the Examples</a>.
 * <p>
 * <li>Compile the example by executing an <font face="Courier New" size = -1>ant</font> 
 * build script or a command.
 * <p>
 * <ul>
 * <li>An <font face="Courier New" size = -1>ant</font> build script is available in the
 * <font face="Courier New" size = -1>samples/examples/jdbc/datasource</font> directory. Enter the following command
 * to execute the build script:
 * <p>
 * <pre><font face="Courier New"><b>ant</b></font></pre>
 * <li>Enter the following command to compile the example:
 * <p>
 * <pre><font face="Courier New"><b>$ javac -d %CLIENT_CLASSES% simplesql.java</b></font></pre> 
 * </ol>
 * <p> 
 * <li>Start WebLogic Server with the <a href=../../examples.html>examples configuration</a>.
 * </ol>
 * <p>
 * <h3>Configure the Example</h3>  
 * <ol>
 * <p><li>Open the <a href=../../examples.html#console>Administration Console</a>.
 * <p><li>Verify that the <a href=../../examples.html#webApp>Examples Web Application</a> has been deployed on your server.
 * <p><li>Verify that the demo datasource, examples-dataSource-demoPool, points to the demoPool connection pool
 * and that the demoPool is enabled, as follows: 
 * <ul>
 * <p>
 * <li>Click to expand the JDBC node in the left-hand pane.
 * <li>Click to expand the Connection Pool node, then verify that demoPool appears under Connection Pool.
 * <li>Click to expand the Data Source node, then verify that examples-dataSource-demoPool shows 
 * demoPool as the pool name in the right-hand pane.</ul>
 * </ul>
 * </ol>
 * <p><h3>Run the Example</h3>
 * <ol>
 * <li>Execute the following command in your development shell: 
 * 
 * <pre><b>$ java examples.jdbc.datasource.simplesql</b></pre>
 * 
 * </ol>
 * <h3>There's more</h3>
 * <p>
 * <ul>
 * <li>For more information on managing datasources and connection pools, see Managing WebLogic JDBC in the Administration Guide.
 * <p>
 * <li>For more information on datasources and connection pools, begin with <a
 * href="http://e-docs.bea.com/wls/docs61/jdbc/intro.html#ov_connpools">Overview of Connection Pools</a> and <a
 * href="http://e-docs.bea.com/wls/docs61/jdbc/intro.html#DataSources">Overview of DataSources</a> in Programming WebLogic JDBC.
 * </ul> 
 * @author Copyright (c) 2000-2001 by BEA Systems, Inc.  All Rights Reserved.
 */



public class simplesql {
  
  public static void main(String argv[])
       throws Exception
  {
    java.sql.Connection conn = null;
    java.sql.Statement stmt  = null;
    try {

      // ============== Make connection to database ==================
      // Obtain a Datasource connection from the WebLogic JNDI tree.
      Context ctx = null;
     
      // Put connection properties in to a hashtable. 
      Hashtable ht = new Hashtable();
      ht.put(Context.INITIAL_CONTEXT_FACTORY,
             "weblogic.jndi.WLInitialContextFactory");
      ht.put(Context.PROVIDER_URL,
             "t3://localhost:7001");

      // Get a context for the JNDI look up
      ctx = new InitialContext(ht);
      javax.sql.DataSource ds
          = (javax.sql.DataSource) ctx.lookup ("examples-dataSource-demoPool");
      conn = ds.getConnection();
      System.out.println("Making connection...\n");
      
      // execute some SQL statements to demonstrate the connection. 
      stmt = conn.createStatement();
    
      try {
        stmt.execute("drop table empdemo");
        System.out.println("Table empdemo dropped.");
      } catch (SQLException e) {
          System.out.println("Table empdemo doesn't need to be dropped.");
      }
    
      stmt.execute("create table empdemo (empid int, name varchar(30), dept int)");
      System.out.println("Table empdemo created.");
    
      int numrows = stmt.executeUpdate("insert into empdemo values (0, 'John Smith', 12)");
      System.out.println("Number of rows inserted = " + numrows);
    
      numrows = stmt.executeUpdate("delete from empdemo where empid = 0");
      System.out.println("Number of rows deleted = " + numrows);
    } catch (Exception e) {
        System.out.println("Exception was thrown: " + e.getMessage());
    } finally {
        try {
          if (stmt != null)
            stmt.close();
          if (conn != null)
            conn.close();
        } catch (SQLException sqle) {
            System.out.println("SQLException during close(): " + sqle.getMessage());
        }
     }
  }
    
}





