package examples.jdbc.mssqlserver4;

import java.io.*;
import java.sql.*;
import java.util.Properties;

/** This example creates <i>n</i> database connections and runs
 * <i>n</i> threads on each.  You need valid licenses for the number
 * of connections you want to try. The trial version provides 2
 * connections only. Contact support@weblogic.com for more connection
 * licenses.  Your SQL Server must also be configured to accept the
 * number of connections you want to create.
 * <p>
 * The threads run for a designated number of seconds, executing a
 * SQL SELECT statement. You can modify the number of seconds and the
 * SQL command in the code.
 * <p>
 * <h3>Build the Example</h3>  
 * <ol>
 * <li> Open a new command shell. 
 * <p><li>Set up this development shell as described in 
 * <a href=../../examples.html#environment>Setting up Your Environment for 
 * Building and Running the Examples</a>.
 * <p>
 * <li>Change your CLASSPATH if neccessary, depending on which version of WebLogic jDriver for Microsoft SQL Server
 * you are using. If you are using Version 7.0 you do not need to make changes. If you are using
 * Version 6.5, pre-pend your classpath as follows by adding the jar file: 
 * <pre>  $ <b>set CLASSPATH=%WL_HOME%/lib/mssqlserver4v65.jar;%CLASSPATH%</b></pre>
 * <p>
 * <li> Change the database URL, login, and password connection parameters in the example source code 
 * to correspond to your Microsoft SQL Server configuration.
 * If you need more help, check the section on connecting
 * to a database in the programming guide, <a
 * href="http://e-docs.bea.com/wls/docs61/mssqlserver4/API_jmsq4.html">Using WebLogic jDriver for Microsoft SQL Server</a>.
 * <p>
 * <li>Change the SQL <b>Select</b> statement to use an existing table on your SQL Server.
 * <p>
 * <li>Compile this example using the following command line:
 * <pre>  $ <b>javac -d %CLIENT_CLASSES% ThreadsTests.java</b></pre>
 * 
 * </ol>
 * <p>
 * <h3>Run the Example</h3>
 * <ol>
 * <li>Execute the following command in your development shell: 
 * 
 * <pre><b>$ java examples.jdbc.mssqlserver4.ThreadsTests</b></pre>
 * 
 * </ol>
 * <h3>There's More</h3>
 *
 * For more information about the WebLogic jDriver for Microsoft SQL Server, see   <a
 * href="http://e-docs.bea.com/wls/docs61/mssqlserver4/API_jmsq4.html">Using WebLogic jDriver for Microsoft SQL Server</a>. 
 * <p> * 
 * @author Copyright (c) 1998-2001 by BEA Systems, Inc. All Rights Reserved. 
 */

public class ThreadsTests
{
  static String driverName = "weblogic.jdbc.mssqlserver4.Driver"; 

  // Number of different connections to the server.  When you download
  // the trial software on the net it comes with a single connection
  // license. If you want to run this test you need a multi-connection
  // license.

  static int kNumberOfConnections  = 2;                                      

  // Number of threads sharing the same connection. You can have as
  // many as your computer and database will take.  All the threads
  // use the same connection at the same time. There is no need for
  // them to be synchronized with one another since the driver takes
  // care of synchronization/caching.
  static int kThreadsPerConnection = 3;                             

  // How long should this test last (in seconds)
  static int kTestSeconds = 60;                               

  public static void main(String [] argv)
  {      
      String user = "sa";
      String password = "secret";
      String server = "myHost:1433";
      String url = "jdbc:weblogic:mssqlserver4";  
      String query = "select * from publishers";     

      try {
        for (int i = 0; i < argv.length; i++) 
        {
          if (argv[i].equals("-user")) {
            i++; 
            user = (argv[i].equals("null") ? "" : argv[i]);
          } 
          else if (argv[i].equals("-password")) {
            i++; 
            password = (argv[i].equals("null") ? "" : argv[i]);
          }
          else if (argv[i].equals("-server")) {
            i++; 
            server = (argv[i].equals("null") ? "" : argv[i]);
          }
          else if (argv[i].equals("-url")) {
            i++; 
            url = (argv[i].equals("null") ? "" : argv[i]);
          }
          else if (argv[i].equals("-query")) {
            i++; 
            query = (argv[i].equals("null") ? "" : argv[i]);
          }
        }
      } catch(ArrayIndexOutOfBoundsException aiobe) {
        System.err.println("\nUsage: java examples.jdbc.mssqlserver4.ThreadsTests [options] \n\n" +
                           "where options include:\n" +
                           "    -user <user>            User name to be passed to database.\n" +
                           "    -password <password>    User password to be passed to database.\n" +
                           "    -server <server>        DNS name of database server.\n" +
                           "    -url <url>              URL of database.\n" +
                           "    -query <query>          SQL query.\n");
        System.exit(1);
      }

      Properties props = new Properties();
      props.put("user", user);
      props.put("password", password);
      props.put("server", server);

      Driver myDriver = null;
      java.sql.Connection conn = null;
      
      try
      {
        // Load the driver in the Java VM.
        myDriver = (Driver) Class.forName(driverName).newInstance();

        ReadingThread readers [] = 
          new ReadingThread[kNumberOfConnections * kThreadsPerConnection];

        System.out.println("ThreadsTests - creating " + 
                           (kNumberOfConnections * kThreadsPerConnection) + 
                           " threads using " + kNumberOfConnections + 
                           " connections to the server...\n");

        for (int i = 0 ; i < kNumberOfConnections ; i++)
          {
            //  Connect to the database as administrator (you may need
            //  to type your login/password here) 
            conn = myDriver.connect(url, props);

            // Create a few threads all using the same connection
            for (int j = 0 ; j < kThreadsPerConnection ; j++)
              {
                readers[i * kThreadsPerConnection + j] = new ReadingThread(conn, query);
              }
          }

        System.out.println("ThreadsTests - " + 
                           (kNumberOfConnections * kThreadsPerConnection) + 
                           " threads were created, starting them...\n");

        // Scan all the threads we launched
        for (int i = 0 ; i < readers.length ; i++)
          {
            if (readers[i] != null) 
              readers[i].start();                      // Start threads
          }

        System.out.println("ThreadsTests - main thread is going to sleep so " +
                           "readers can torture the server...\n");

        Thread.sleep(kTestSeconds * 1000); // Sleep for a while (let
                                  // the threads torture the server)

        System.out.println("ThreadsTests - notifying readers that it's time to quit...\n");

        for(int i = 0 ; i < readers.length ; i++) // Scan all the
                           // threads we launched
          {
            if(readers[i] != null) readers[i].running = false; // Time
              // to quit bothering the server
          }

        for(int i = 0 ; i < readers.length ; i++) // Scan all the
                           // threads we launched
          {
            if(readers[i] != null && readers[i].isAlive()) // If the
                  // i-th thread is still alive
              {
                try { readers[i].join(); } // Wait for this reader to
                                  // die

                catch(InterruptedException ie) { }
              }

            readers[i] = null;
          }

        System.out.println("\nThreadsTests - all readers stopped, we ran " + 
                           totalRuns + " times (read " + totalRows + 
                           " rows total).");
      } catch(Exception sqlex) {
          System.out.println("ThreadsTests - exception " + sqlex);
      } finally { //close connections in a finally block
          try {
            if (conn != null)
              conn.close();
           } catch (SQLException sqle) {
               System.out.println("SQLException was thrown: " + sqle.getMessage());
           }
      }
  }

  static int totalRuns = 0; static int totalRows = 0;

  synchronized static void updateStatistics(int runs, int rows)
  {
    totalRuns += runs; 
    totalRows += rows;
  }
}

/** The threads created by ThreadTests execute this class. You can
 * change the SQL the threads execute by changing the argument to
 * statement.executeQuery.
 * 
 * @param connection, the connection the thread uses.
 * @author Copyright (c) 1998-2000 by BEA Systems, Inc. All Rights Reserved.  */
class ReadingThread extends Thread
{
    ReadingThread(Connection connection, String query)
    {
        this.connection = connection;
        if (query != null && query.length() != 0) {
          this.query = query;
        }
    }

    String query = "select * from publishers";
 
    Connection connection = null; // Connection used by this thread

    boolean running = true; // True while we're running

    public void run()
    {
        int runs = 0, rows = 0;

        for( ; running ; runs++) // Keep repeating the query
        {
            Statement statement = null;

            try
            {
                // Create a statement that returns some table. You may
                // want to do something different here.
                statement = connection.createStatement(); 

                // You can put any SQL statement you want here...
                ResultSet result = 
                  statement.executeQuery(query);

                // Number of columns per row
                int i = 0, columns = result.getMetaData().getColumnCount();     

                // Scan all rows in the result set
                for( ; result.next() && running ; i++)                          
                  {
                    // Scan all columns in the row
                    for(int j = 1 ; j <= columns ; j++)
                    {
                         // Get the value as a string (any type will work this way)
                        String s = result.getString(j);
                        try
                        {
                            Thread.sleep(25);
                        }
                        catch(InterruptedException ie) { }
                    }
                }

                System.out.println(" read " + i + " rows [" + this + "]...");
                ThreadsTests.updateStatistics(1,i); rows += i;
            }

            catch(SQLException sqlEx) 
              { System.out.println(sqlEx); }

            if(statement != null)
            {
                try { statement.close(); } 
                catch (SQLException sqlEx) 
                  { System.out.println(sqlEx); }
            }
        }

        // We can't close the connection because it may be in use by
        // other threads
        connection = null; 
        System.out.println(toString() + " - ran the test " + runs 
                           + " times (total of " + rows + " rows read).");
    }
}

