package examples.jdbc.mssqlserver4;

import java.io.*;
import java.util.*;
import java.net.*;
import java.sql.*;

/**
 * This simple example shows how JDBC Extended SQL can be embedded
 * in standard SQL sent to the DBMS. Extended SQL is part of the
 * JDBC spec, and provides a DBMS-neutral way of using the many
 * secondary functions provided by any DBMS, but in a proprietary
 * syntax. The WebLogic JDBC drivers will convert extended SQL syntax
 * to the DBMS-specific syntax for the relevant function. Please see
 * our documentation for links to the complete Extended SQL specification.
 * <p>
 * This example is run on the command line and is output to
 * System.out.
 * <p>
 * Please note that this example uses a simple method for obtaining
 * result sets, update counts, and/or the output parameter values it may
 * generate. This is always possible with simple, known queries.
 * When executing complex or unknown possibly multi-statement SQL
 * or stored procedures, you should use the example in <tt><a href="complexproc.java">complexproc.java</a></tt>
 * to ensure that all results become available.
 * <p><h3>Build the Example</h3>  
 * <ol>
 * <li> Open a new command shell. 
 * <p><li>Set up this development shell as described in 
 * <a href=../../examples.html#environment>Setting up Your Environment for 
 * Building and Running the Examples</a>.
 * <p>
 * <li>Change your CLASSPATH if neccessary, depending on which version of WebLogic jDriver for Microsoft SQL Server
 * you are using. If you are using Version 7.0 you do not need to make changes. If you are using
 * Version 6.5, pre-pend your classpath as follows by adding the jar file: 
 * <pre>  $ <b>set CLASSPATH=%WL_HOME%/lib/mssqlserver4v65.jar;%CLASSPATH%</b></pre>
 * <p>
 * <li> Change the database URL, login, and password connection parameters defined in the
 * example source code to correspond to your Microsoft SQL Server configuration.
 * If you need more help, check the section on connecting
 * to a database in the programming guide, <a
 * href="http://e-docs.bea.com/wls/docs61/mssqlserver4/API_jmsq4.html">Using WebLogic jDriver for Microsoft SQL Server</a>.
 * <p>
 * <li>Compile this example using the following command line:
 * <pre>  $ <b>javac -d %CLIENT_CLASSES% exsql.java</b></pre>
 * 
 * </ol>
 * <p><h3>Run the Example</h3>
 * <ol>
 * <li>Execute the following command in your development shell: 
 * 
 * <pre><b>$ java examples.jdbc.mssqlserver4.exsql</b></pre>
 * 
 * </ol>
 * <h3>There's More</h3>
 *
 * For more information about the WebLogic jDriver for Microsoft SQL Server, see   <a
 * href="http://e-docs.bea.com/wls/docs61/mssqlserver4/API_jmsq4.html">Using WebLogic jDriver for Microsoft SQL Server</a>. 
 * <p>
 * @author         Copyright (c) 1997-2001 by BEA Systems, Inc. All Rights Reserved.
 */

public class exsql
{
  public static void main(String argv[])
  {
    String user = "sa";
    String password = "secret";
    String server = "myHost:1433";
    String url = "jdbc:weblogic:mssqlserver4";       

    try {
      for (int i = 0; i < argv.length; i++) 
      {
        if (argv[i].equals("-user")) {
          i++; 
          user = (argv[i].equals("null") ? "" : argv[i]);
        } 
        else if (argv[i].equals("-password")) {
          i++; 
          password = (argv[i].equals("null") ? "" : argv[i]);
        }
        else if (argv[i].equals("-server")) {
          i++; 
          server = (argv[i].equals("null") ? "" : argv[i]);
        }
        else if (argv[i].equals("-url")) {
          i++; 
          url = (argv[i].equals("null") ? "" : argv[i]);
        }
      }
    } catch(ArrayIndexOutOfBoundsException aiobe) {
      System.err.println("\nUsage: java examples.jdbc.mssqlserver4.exsql [options] \n\n" +
                         "where options include:\n" +
                         "    -user <user>            User name to be passed to database.\n" +
                         "    -password <password>    User password to be passed to database.\n" +
                         "    -server <server>        DNS name of database server.\n" +
                         "    -url <url>              URL of database.\n");
      System.exit(1);
    }

    java.sql.Connection conn = null;
    java.sql.Statement stmt  = null;
    java.sql.ResultSet rs    = null;
    Driver myDriver = null;

    Properties props = new Properties();
    props.put("user", user);
    props.put("password", password);
    props.put("server", server);

    try
      {
	myDriver = (Driver)Class.forName("weblogic.jdbc.mssqlserver4.Driver").newInstance();
	conn = myDriver.connect("jdbc:weblogic:mssqlserver4", props );
        stmt = conn.createStatement();
	
        // This is an example of SQL including a few extended SQL escapes.
        // An extended SQL element is enclosed by {}. This example uses
        // the function dayname, the function now(), and a date, time,
        // and timestamp escape. Note that they can be nested.
	
        String exsql = "select {fn dayname {fn now()}},   -- Today    \n" +
	  "           {d '1997-05-24'},  -- a date                    \n" +
	  "           {t '10:30:29' },  -- a time                     \n" +
	  "           {ts '1997-05-24 10:30:29.123'}, -- a timestamp  \n" +
	  "          '{string data with { or } will not be altered'   \n" +
	  "--  Also note that you can safely include } and { in comments";
	
        System.out.println(" The Extended SQL:\n" + exsql + "\n\n becomes:\n");
        System.out.println( conn.nativeSQL( exsql ));
	
        rs = stmt.executeQuery( exsql );
	
        while (rs.next())
          System.out.println("Got:\n " + rs.getString(1) +
                               "\n "   + rs.getString(2) +
                               "\n "   + rs.getString(3) +
                               "\n "   + rs.getString(4) +
                               "\n "   + rs.getString(5) );
      } catch (Exception e) {
	  e.printStackTrace();
      } finally {
	  try {
            if (rs != null)
              rs.close();
            if (stmt != null)
              stmt.close();
            if (conn != null)
              conn.close();
          } catch (SQLException sqle) {
              System.out.println("SQLException was thrown: " + sqle.getMessage());
          }
      }
  }
}

