package examples.jdbc.mssqlserver4;

import java.sql.*;
import java.util.Properties;

/**
 * This simple example shows how JDBC can be used to execute DDL and
 * DML.
 * <p>
 * This example is run on the command line and is output to
 * System.out.
 * <p>
 * Please note that this example uses a simple method for obtaining
 * result sets, update counts, and/or the output parameter values it may
 * generate. This is always possible with simple, known queries.
 * When executing complex or unknown possibly multi-statement SQL
 * or stored procedures, you should use the example in <tt><a href="complexproc.java">complexproc.java</a></tt>
 * to ensure that all results become available.
 * <P>
 * <p><h3>Build the Example</h3>  
 * <ol>
 * <li> Open a new command shell. 
 * <p><li>Set up this development shell as described in 
 * <a href=../../examples.html#environment>Setting up Your Environment for 
 * Building and Running the Examples</a>.
 * <p>
 * <li>Change your CLASSPATH if neccessary, depending on which version of WebLogic jDriver for Microsoft SQL Server
 * you are using. If you are using Version 7.0 you do not need to make changes. If you are using
 * Version 6.5, pre-pend your classpath as follows by adding the jar file: 
 * <pre>  $ <b>set CLASSPATH=%WL_HOME%/lib/mssqlserver4v65.jar;%CLASSPATH%</b></pre>
 * <p>
 * <li> Change the database URL, login, and password connection parameters defined in the example source
 * code to correspond to your Microsoft SQL Server configuration.
 * If you need more help, check the section on connecting
 * to a database in the programming guide, <a
 * href="http://e-docs.bea.com/wls/docs61/mssqlserver4/API_jmsq4.html">Using WebLogic jDriver for Microsoft SQL Server</a>.
 * <p>
 * <li>Compile this example using the following command line:
 * <pre>  $ <b>javac -d %CLIENT_CLASSES% records.java</b></pre>
 * 
 * </ol>
 * <p><h3>Run the Example</h3>
 * <ol>
 * <li>Execute the following command in your development shell: 
 * 
 * <pre><b>$ java examples.jdbc.mssqlserver4.records</b></pre>
 * 
 * </ol>
 * <h3>There's More</h3>
 *
 * For more information about the WebLogic jDriver for Microsoft SQL Server, see   <a
 * href="http://e-docs.bea.com/wls/docs61/mssqlserver4/API_jmsq4.html">Using WebLogic jDriver for Microsoft SQL Server</a>. 
 * <p>
 * @author         Copyright (c) 1997-2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class records {
  
  public static void main(String argv[])
  {
    String user = "sa";
    String password = "secret";
    String server = "myHost:1433";
    String url = "jdbc:weblogic:mssqlserver4";       

    try {
      for (int i = 0; i < argv.length; i++) 
      {
        if (argv[i].equals("-user")) {
          i++; 
          user = (argv[i].equals("null") ? "" : argv[i]);
        } 
        else if (argv[i].equals("-password")) {
          i++; 
          password = (argv[i].equals("null") ? "" : argv[i]);
        }
        else if (argv[i].equals("-server")) {
          i++; 
          server = (argv[i].equals("null") ? "" : argv[i]);
        }
        else if (argv[i].equals("-url")) {
          i++; 
          url = (argv[i].equals("null") ? "" : argv[i]);
        }
      }
    } catch(ArrayIndexOutOfBoundsException aiobe) {
      System.err.println("\nUsage: java examples.jdbc.mssqlserver4.records [options] \n\n" +
                         "where options include:\n" +
                         "    -user <user>            User name to be passed to database.\n" +
                         "    -password <password>    User password to be passed to database.\n" +
                         "    -server <server>        DNS name of database server.\n" +
                         "    -url <url>              URL of database.\n");
      System.exit(1);
    }

    java.sql.Connection conn         = null;
    java.sql.Statement stmt          = null;
    java.sql.PreparedStatement pstmt = null;
    java.sql.ResultSet rs            = null;
    
    try {
      Properties props = new Properties();
      props.put("user", user);
      props.put("password", password);
      props.put("server", server);

      Driver myDriver = (Driver)
          Class.forName("weblogic.jdbc.mssqlserver4.Driver").newInstance();
      conn = myDriver.connect("jdbc:weblogic:mssqlserver4", props);
      stmt = conn.createStatement();
    
      // Create temporary table
      stmt.execute("create table #empdemo(empid int, name char(30), dept int)");
    
      // Insert 10 records
      System.out.println("Inserting 10 records...");
      String inssql = "insert into #empdemo(empid, name, dept) values (?, ?, ?)";
      pstmt = conn.prepareStatement(inssql);
    
      for (int i = 0; i < 10; i++) {
        pstmt.setInt(1, i);
        pstmt.setString(2, "Person " + i);
        pstmt.setInt(3, i);
        pstmt.execute();
      }
      
      pstmt.close();
    
      // Update 10 records
      System.out.println("Updating 10 records...");
      String updsql = "update #empdemo set dept = dept + ? where empid = ?";
      pstmt = conn.prepareStatement(updsql);
    
      for (int i = 0; i < 100; i++) {
        pstmt.setInt(1, i);
        pstmt.setInt(2, i);
        pstmt.execute();
      }
    
      // Get the current contents and print them out
      rs = stmt.executeQuery("select * from #empdemo");
      while (rs.next())
      { 
        System.out.println( rs.getInt(1) + ", "
                       + rs.getString(2) + ", "
                       + rs.getInt(3));
      }
    } catch (Exception e) {
        System.out.println("Exception was thrown: " + e.getMessage());
    } finally { //alwawy close connections in a finally block
        try {
          if (stmt != null)
            stmt.close();
          if (pstmt != null)
            pstmt.close();
          if (conn != null)
            conn.close();
        } catch (SQLException sqle) {
            System.out.println("SQLException was thrown: " + sqle.getMessage());
        }
    }
  }
}

