package examples.jdbc.oracle;
import java.sql.*;
import java.io.*;
import weblogic.jdbc.common.*;
import weblogic.jdbc.common.*;
import java.util.Properties;

/**
 * This example demonstrates the use of Oracle Blob and Clob datatypes.
 * 
 * <p><h3>Build the Example</h3>  
 * <ol>
 * <li> Open a new command shell. 
 * <p><li>Set up this development shell as described in 
 * <a href=../../examples.html#environment>Setting up Your Environment for 
 * Building and Running the Examples</a>.
 * <p>
 * <ul><li>There are two pathnames required in your environment
 * 1) the directory in which the WebLogic shared library (native interface) resides, and
 * 2) the directory in which the vendor-supplied libraries from Oracle
 * reside (<i>bin</i> for NT or <i>lib</i> for UNIX). 
 * Be sure to include these pathnames in your path.
 * <p>
 * <li> For complete information, see <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#environment">Setting Up the Environment for Using 
 * WebLogic jDriver for Oracle</a>
 * in <i>Installing and Using WebLogic jDriver for Oracle</i></a>. 
 * The following sections contain instructions for specific platforms:
 * <p>
 * <ul>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#nt">Windows NT</a>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#solaris">Solaris</a>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#hp">HP</a>
 * </ul>
 * <p></ul>
 * <li> This example also requires the following:
 * <p><ul>
 * <li> An Oracle server version 8
 * <li> Oracle client libraries version 8.0.5 or later
 * </ul>
 * <p>
 * <li> Change connection parameters to correspond to your Oracle configuration.
 * If you need more help, check the section on connecting
 * to a database in the programming guide, <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/API_joci.html">Using WebLogic jDriver for Oracle</a>.
 * <p>
 * <li>Compile the example by executing the following command or by executing the <a href=../../examples.html#buildScripts>build script</a> 
 * provided for this example in the <font face="Courier New" size = -1>samples/examples/jdbc/oracle</font> 
 * directory. The script will perform the following step:
 * <p> 
 * <ul>
 * <li>Compile the OracleBlobClob class as shown in this example for <b>Windows NT/2000</b>:
 * <p>
 * <pre>  $ <b>javac -d %CLIENT_CLASSES% OracleBlobClob.java</b></pre>
 * </ol>
 * <p><h3>Run the Example</h3>
 * <ul>
 * <li>Execute the following command in your development shell: 
 * 
 * <pre><b>$ java examples.jdbc.oracle.OracleBlobClob</b></pre>
 * 
 * </ul>
 * <h3>There's More</h3>
 *
 * For more information about the WebLogic jDriver for Oracle, see   <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/index.html"><i>Installing and Using WebLogic jDriver for Oracle</i></a>. 
 * <p>
 * @author Copyright (c) 2000-2001 by BEA Systems, Inc.  All Rights Reserved.
 */

public class OracleBlobClob {
  
  public static void main(String argv[])
  {
    String user = "scott";
    String password = "tiger";
    String server = "DEMO";

    try {
      for (int i = 0; i < argv.length; i++) 
      {
        if (argv[i].equals("-user")) {
          i++; 
          user = (argv[i].equals("null") ? "" : argv[i]);
        } 
        else if (argv[i].equals("-password")) {
          i++; 
          password = (argv[i].equals("null") ? "" : argv[i]);
        }
        else if (argv[i].equals("-server")) {
          i++; 
          server = (argv[i].equals("null") ? "" : argv[i]);
        }
      }
    } catch(ArrayIndexOutOfBoundsException aiobe) {
      System.err.println("\nUsage: java examples.jdbc.oracle.OracleBlobClob [options] \n\n" +
                         "where options include:\n" +
                         "    -user <user>            User name to be passed to database.\n" +
                         "    -password <password>    User password to be passed to database.\n" +
                         "    -server <server>        DNS name of database server.\n"); 
      System.exit(1);
    }

    java.sql.Blob myBlob     = null;
    java.sql.Clob myClob     = null;
    java.sql.Connection conn = null;

    // get a connection to the Oracle DBMS
    // substitute the name of the machine hosting your 
    //  Oracle server for myOracle8Server

    Properties props = new Properties();
    props.put("user",     user);
    props.put("password", password);
    props.put("server",   server);

    try {
      Driver myDriver = (Driver)
          Class.forName("weblogic.jdbc.oci.Driver").newInstance();
      conn = myDriver.connect("jdbc:weblogic:oracle" , props);
      

      // set Oracle's Auto Commit feature to false. 
      // This is necessary when manipulating Blobs and Clobs.
      conn.setAutoCommit(false);

      // ============== Create Table ==================
      // Create a table with a Blob and Clob column
      try {
          // if table does not exist, create it. 
          Statement crstmt = conn.createStatement();
          System.out.println("\nCreating table with Blobs and Clobs...");
          crstmt.execute("create table lobtest (id int, blobcol Blob, clobcol Clob)");
          crstmt.close();
      }
      catch (Exception e) {
          System.out.println("Exception: " + e);
          System.out.println("Table already exists. Dropping it and re-creating...");
          Statement crstmt2 = conn.createStatement();
          crstmt2.execute("drop table lobtest");
          crstmt2.execute("create table lobtest (id int, blobcol Blob, clobcol Clob)");
          crstmt2.close();
      }
      System.out.println("Table created.");


      // ============== Initializing blob and clob values ==================
      Statement stmt = conn.createStatement();
      System.out.println("\nInserting row with blank blob and clob columns...");
      stmt.execute("insert into lobtest values (44,EMPTY_BLOB(),EMPTY_CLOB())");
      System.out.println("Row has been inserted.");

      // ============== Manipulating the Blob column ======================
      // get a reference to the Blob column
      stmt.execute("select * from lobtest where id=44");
      ResultSet rs = stmt.getResultSet();
      while ( rs.next() ) {
          myBlob = rs.getBlob("blobcol");
      }

      // Create a byte array and store some data in it
      System.out.println("\nCreating the following byte array:");
      int STREAM_SIZE = 10;
      byte[] b = new byte[STREAM_SIZE];
      for (int i=0; i < STREAM_SIZE; i++) {
          b[i] = (byte)(40 + (i%20)); // range 40-60
          System.out.println("byte[" + i + "] = " + b[i]);         
      }

      // Write the byte array to a stream and store it in the Blob column
      System.out.println
          ("\nWriting the byte array to a stream" +
           " and storing it in the table as a blob...");
      InputStream is = new ByteArrayInputStream(b);
      java.io.OutputStream os =
          ((weblogic.jdbc.common.OracleBlob) myBlob).getBinaryOutputStream();
      byte[] inBytes = new byte[STREAM_SIZE]; 
      int numBytes = is.read(inBytes);
      
      // write the input stream to the output stream
      while (numBytes > 0) {
          os.write(inBytes, 0, numBytes);
          numBytes = is.read(inBytes);
      }
      // The flush() method causes the data to be written to the table
      os.flush();

      //  read back the blob
      System.out.println("\nReading the blob back from the table and displaying:");
      Statement readblob = conn.createStatement();
      readblob.execute("select * from lobtest where id=44");
      ResultSet rsreadblob = readblob.getResultSet();

      // read the blob into a byte array and display
      byte[] r = new byte[STREAM_SIZE];
      while ( rsreadblob.next() ) { 
          Blob myReadBlob =  rsreadblob.getBlob("blobcol");
          java.io.InputStream readis = myReadBlob.getBinaryStream();
          for (int i=0 ; i < STREAM_SIZE ; i++) {
              r[i] = (byte) readis.read();
              System.out.println("output [" + i + "] = " + r[i]);
          }
      }
      
      
      // create some character data to work with
      String ss = "abcdefghijklmnopqrstuvwxyz";
      System.out.println("\nCreated the following string to be stored as a clob:\n" +
                         ss);
      
      // ============== Manipulating the Clob column ======================
      // get a reference to the clob column
      stmt.execute("select * from lobtest where id=44");
      ResultSet crs = stmt.getResultSet();
      while ( crs.next() ) {
          myClob = crs.getClob("clobcol");
          
          java.io.OutputStream osss =
              ((weblogic.jdbc.common.OracleClob) myClob).getAsciiOutputStream();
          byte[] bss = ss.getBytes("ASCII");
          osss.write(bss);
          osss.flush();
      }
      conn.commit();
      
      // read back the clob
      System.out.println("\nReading the clob back from the table and displaying:");
      Statement readclob = conn.createStatement();
      readclob.execute("select * from lobtest where id=44");
      ResultSet rsreadclob = readclob.getResultSet();
            
      // read the clob in as and ASCII stream, write to a character array, and display
      while ( rsreadclob.next() ) { 
          Clob myReadClob =rsreadclob.getClob("clobcol");    
          java.io.InputStream readClobis = myReadClob.getAsciiStream();
          char[] c = new char[26];
          for (int i=0 ; i < 26  ; i++) {
              c[i] = (char) readClobis.read();
              System.out.println("output [" + i + "] = " + c[i]);
          }
      }        
                
      // Drop the table and clean up connections
      System.out.println("\nDropping table...");
      Statement dropstmt = conn.createStatement();
      dropstmt.execute("drop table lobtest");
      System.out.println("Table dropped.");
      
    } catch (Exception e) {
        System.out.println("Exception was thrown: " + e.getMessage());
    } finally {
        try {
          if (conn != null)
            conn.close();
        } catch (SQLException sqle) {
            System.out.println("SQLException was thrown: " + sqle.getMessage());
        }
    }
  }
    
}

