package examples.jdbc.oracle;
import java.sql.*;
import java.util.Properties;

/**
 * This example demonstrates the use of the JDBC 2.0 Batch Update feature with an Oracle database.
 * <p>
 * <h3>Build the Example</h3>
 * <ol>
 * <li> Open a new command shell.
 * <p><li>Set up this development shell as described in
 * <a href=../../examples.html#environment>Setting up Your Environment for
 * Building and Running the Examples</a>.
 * <p>
 * <ul><li>There are two pathnames required in your environment
 * 1) the directory where the WebLogic shared library (native interface) is located, and
 * 2) the directory in which the vendor-supplied libraries from Oracle reside (<i>bin</i> for NT or <i>lib</i> for UNIX).
 * Be sure to include these pathnames in your path.
 * <p>
 * <li> For complete information, see <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#environment">Setting Up the Environment for Using WebLogic j4 for Oracle</a>
 * in <i>Installing and Using WebLogic jDriver for Oracle</i></a>.
 * The following sections contain instructions for specific platforms:
 * <p>
 * <ul>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#nt">Windows NT</a>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#solaris">Solaris</a>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#hp">HP</a>
 * </ul>
 * <p></ul>
 * <li> Change connection parameters to correspond to your Oracle configuration.
 * If you need more help, check the section on connecting
 * to a database in the programming guide, <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/API_joci.html">Using WebLogic jDriver for Oracle</a>.
 * <p>
 * <li>Compile the example by executing an <font face="Courier New" size = -1>ant</font> 
 * build script or a command.
 * <p>
 * <ul>
 * <li>An <font face="Courier New" size = -1>ant</font> build script is available in the
 * <font face="Courier New" size = -1>samples/examples/jdbc/oracle</font> directory. Enter the following command
 * to execute the build script:
 * <p>
 * <pre><font face="Courier New">  $ <b>ant</b>, or</font></pre> 
 * <p>
 * <li>Compile the batchUpdate class as shown in this example for <b>Windows NT/2000</b>:
 * <p>
 * <pre>  $ <b>javac -d %CLIENT_CLASSES% batchUpdate.java</b></pre>
 * </ol>
 * <p>
 * <h3>Run the Example</h3>
 * <ul>
 * <li>Execute the following command in your development shell:
 *
 * <pre><b>$ java examples.jdbc.oracle.batchUpdate</b></pre></ul>
 * </li>
 *
 * <h3>There's More</h3>
 *
 * For more information about the WebLogic jDriver for Oracle, see   <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/index.html"><i>Installing and Using WebLogic jDriver for Oracle</i></a>.
 * <p>
 * @author Copyright (c) 2000-2001 by BEA Systems, Inc.  All Rights Reserved.
 */

public class batchUpdate {

  public static void main(String argv[])
  {
    String user = "scott";
    String password = "tiger";
    String server = "DEMO";

    try {
      for (int i = 0; i < argv.length; i++) 
      {
        if (argv[i].equals("-user")) {
          i++; 
          user = (argv[i].equals("null") ? "" : argv[i]);
        } 
        else if (argv[i].equals("-password")) {
          i++; 
          password = (argv[i].equals("null") ? "" : argv[i]);
        }
        else if (argv[i].equals("-server")) {
          i++; 
          server = (argv[i].equals("null") ? "" : argv[i]);
        }
      }
    } catch(ArrayIndexOutOfBoundsException aiobe) {
      System.err.println("\nUsage: java examples.jdbc.oracle.batchUpdate [options] \n\n" +
                         "where options include:\n" +
                         "    -user <user>            User name to be passed to database.\n" +
                         "    -password <password>    User password to be passed to database.\n" +
                         "    -server <server>        DNS name of database server.\n"); 
      System.exit(1);
    }

    java.sql.Connection conn = null;
    java.sql.Statement stmt  = null;
    java.sql.ResultSet rs    = null;
    try {
      Properties props = new Properties();
      props.put("user",     user);
      props.put("password", password);
      props.put("server",   server);

      Driver myDriver = (Driver) Class.forName("weblogic.jdbc.oci.Driver").newInstance();
      conn = myDriver.connect("jdbc:weblogic:oracle", props);


      // create a table
      try {
          // if table does not exist, create it.
          stmt = conn.createStatement();
          System.out.println("\nCreating table ...");
          stmt.execute
              ("create table batchTest (id VARCHAR(10), col2 NUMBER, col3 NUMBER)");
      }
      catch (Exception e) {
          System.out.println("Exception: " + e);
          System.out.println("Table already exists. Dropping it and re-creating...");
          stmt.execute("drop table batchTest");
          stmt.execute
              ("create table batchTest (id VARCHAR(10), col2 NUMBER, col3 NUMBER)");
      }
      System.out.println("Table created.");

      // create a Statement object

      // add SQL statements to the batch
      stmt.addBatch("INSERT INTO batchTest VALUES ('JOE', 20,35)");
      stmt.addBatch("INSERT INTO batchTest VALUES ('Bob', 30,44)");
      stmt.addBatch("INSERT INTO batchTest VALUES ('Ed',  34,22)");


      // send the statements to the DBMS in a try/catch block
      // to catch exceptions
      try {
          stmt.executeBatch();
      } catch (Exception e) {
          System.out.println("Exception while executing batch:\n " + e);
      }

      //display the updated rows
      stmt.execute("SELECT * FROM batchTest");
      rs =  stmt.getResultSet();

      while (rs.next() ) {
          System.out.println(rs.getString("id") + " "
                             + rs.getInt("col2") + " " +
                             rs.getInt("col3"));
      }

      // Drop the table and clean up connections
      System.out.println("\nDropping table...");
      stmt.execute("drop table batchTest");
      System.out.println("Table dropped.");
    } catch (Exception e) {
        System.out.println("Exception was thrown: " + e.getMessage());
    } finally { //close connections and statement in a finally block
        try {
          if (stmt != null)
            stmt.close();
          if (rs != null)
            rs.close();
          if (conn != null)
            conn.close();
        } catch (SQLException sqle) {
            System.out.println("SQLException was thrown: " + sqle.getMessage());
        }
    }
  }
}

