package examples.jdbc.oracle;

import java.io.*;
import java.util.*;
import java.net.*;
import java.sql.*;
import java.util.Properties;
import weblogic.common.*;

/**
 * This simple example shows how JDBC Extended SQL can be embedded
 * in standard SQL sent to the DBMS.  Extended SQL is part of the
 * JDBC spec, and provides a DBMS-neutral way of using the many
 * secondary functions that all DBMSes provide, but in a proprietary
 * syntax. The WebLogic JDBC drivers will convert extended SQL syntax
 * to the DBMS-specific syntax for the relevant function. Please see
 * our documentation for links to the complete Extended SQL specification.
 * <p>
 * This example is run on the command line and is output to
 * System.out.
 *
 * <p><h3>Build the Example</h3>  
 * <ol>
 * <li> Open a new command shell. 
 * <p><li>Set up this development shell as described in 
 * <a href=../../examples.html#environment>Setting up Your Environment for 
 * Building and Running the Examples</a>.
 * <p> 
 * <ul><li>There are two pathnames required in your environment
 * 1) the directory in which the WebLogic shared library (native interface) resides, and
 * 2) the directory in which the vendor-supplied libraries from Oracle
 * reside (<i>bin</i> for NT or <i>lib</i> for UNIX). 
 * Be sure to include these pathnames in your path.
 * <p>
 * <li> For complete information, see <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#environment">Setting Up the Environment for Using 
 * WebLogic jDriver for Oracle</a>
 * in <i>Installing and Using WebLogic jDriver for Oracle</i></a>. 
 * The following sections contain instructions for specific platforms:
 * <p>
 * <ul>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#nt">Windows NT</a>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#solaris">Solaris</a>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#hp">HP</a>
 * </ul>
 * <p></ul>
 * <li> Change connection parameters to correspond to your Oracle configuration.
 * If you need more help, check the section on connecting
 * to a database in the programming guide, <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/API_joci.html">Using WebLogic jDriver for Oracle</a>.
 * <p>
 * <li>Compile the example by executing an <font face="Courier New" size = -1>ant</font> 
 * build script or a command.
 * <p>
 * <ul>
 * <li>An <font face="Courier New" size = -1>ant</font> build script is available in the
 * <font face="Courier New" size = -1>samples/examples/jdbc/oracle</font> directory. Enter the following command
 * to execute the build script:
 * <p>
 * <pre><font face="Courier New">  $ <b>ant</b>, or</font></pre> 
 * <p>
 * <li>Compile the exsql class as shown in this example for <b>Windows NT/2000</b>:
 * <p>
 * <pre>  $ <b>javac -d %CLIENT_CLASSES% exsql.java</b></pre>
 * </ol>
 * <p><h3>Run the Example</h3>
 * <ul>
 * <li>Execute the following command in your development shell: 
 * 
 * <pre><b>$ java examples.jdbc.oracle.exsql</b></pre>
 * 
 * </ul>
 * <h3>There's More</h3>
 *
 * For more information about the WebLogic jDriver for Oracle, see   <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/index.html"><i>Installing and Using WebLogic jDriver for Oracle</i></a>. 
 * <p>
 * @author Copyright (c) 1996-2001 by BEA Systems, Inc.  All Rights Reserved.
 */
public class exsql {

  /**
   * In the main, we set properties for the connection, acquire a
   * Connection object, and then create the Extended SQL and
   * execute it in the Statement. The Extended SQL we use here
   * is an example of SQL that includes a few Extended SQL
   * escapes. Note that an Extended SQL element is enclosed by
   * {}. This example uses the function dayname, the function
   * now(), and a date, time, and timestamp escape. Note that
   * Extended SQL escapes can be nested.
   */
  public static void main(String argv[]) throws Exception {

    String user = "scott";
    String password = "tiger";
    String server = "DEMO";

    try {
      for (int i = 0; i < argv.length; i++) 
      {
        if (argv[i].equals("-user")) {
          i++; 
          user = (argv[i].equals("null") ? "" : argv[i]);
        } 
        else if (argv[i].equals("-password")) {
          i++; 
          password = (argv[i].equals("null") ? "" : argv[i]);
        }
        else if (argv[i].equals("-server")) {
          i++; 
          server = (argv[i].equals("null") ? "" : argv[i]);
        }
      }
    } catch(ArrayIndexOutOfBoundsException aiobe) {
      System.err.println("\nUsage: java examples.jdbc.oracle.exsql [options] \n\n" +
                         "where options include:\n" +
                         "    -user <user>            User name to be passed to database.\n" +
                         "    -password <password>    User password to be passed to database.\n" +
                         "    -server <server>        DNS name of database server.\n"); 
      System.exit(1);
    }
    
    Connection conn  = null;
    
    try {
      java.util.Properties props = new java.util.Properties();
      props.put("user",     user);
      props.put("password", password);
      props.put("server",   server);
      
      Driver myDriver = (Driver) Class.forName("weblogic.jdbc.oci.Driver").newInstance();
      conn = myDriver.connect("jdbc:weblogic:oracle", props);

      Statement stmt = conn.createStatement();
      
      String exsql =
        "select {fn dayname ({fn now()})},   -- Today               \n" +
        "           {d '1997-05-24'},  -- a date                    \n" +
        "           {t '10:30:29' },  -- a time                     \n" +
        "           {ts '1997-05-24 10:30:29.123'}, -- a timestamp  \n" +
        "          '{string data with { or } will not be altered'   \n" +
        "           from dual                                       \n" +
        "--{ Also note that you can safely include { and } in comments";
      
      System.out.println(" The Extended SQL:\n" + exsql + "\n\n becomes:\n");
      System.out.println(conn.nativeSQL(exsql));
      
      ResultSet rs = stmt.executeQuery(exsql);
      
      while (rs.next())
        System.out.println("Got: " + rs.getString(1) +
                           "\n "   + rs.getString(2) +
                           "\n "   + rs.getString(3) +
                           "\n "   + rs.getString(4) +
                           "\n "   + rs.getString(5) );
      
    }
    
    catch (Exception e) {
      e.printStackTrace();
    }
    
    finally {
      try {conn.close();}    catch (Exception e) {;}
    }
  }
}
