package examples.jdbc.oracle;

import java.sql.*;
import java.util.Properties;

/**
 * This type of cursor support exists only in Oracle 7.2 
 * and later. Refer to the file <a href="spcursors.sql">spcursors.sql</a> for the
 * PL/SQL that defines these stored procedures.
 *
 * <p><h3>Build the Example</h3>  
 * <ol>
 * <li> Open a new command shell. 
 * <p><li>Set up this development shell as described in 
 * <a href=../../examples.html#environment>Setting up Your Environment for 
 * Building and Running the Examples</a>.
 * <p>
 * <ul><li>There are two pathnames required in your environment
 * 1) the directory in which the WebLogic shared library (native interface) resides, and
 * 2) the directory in which the vendor-supplied libraries from Oracle
 * reside (<i>bin</i> for NT or <i>lib</i> for UNIX). 
 * Be sure to include these pathnames in your path.
 * <p>
 * <li> For complete information, see <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#environment">Setting Up the Environment for Using 
 * WebLogic jDriver for Oracle</a>
 * in <i>Installing and Using WebLogic jDriver for Oracle</i></a>. 
 * The following sections contain instructions for specific platforms:
 * <p>
 * <ul>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#nt">Windows NT</a>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#solaris">Solaris</a>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#hp">HP</a>
 * </ul>
 * <p></ul>
 * <li> Change connection parameters to correspond to your Oracle configuration.
 * If you need more help, check the section on connecting
 * to a database in the programming guide, <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/API_joci.html">Using WebLogic jDriver for Oracle</a>.
 * <p>
 * <li>Compile the example by executing the following command or by executing the <a href=../../examples.html#buildScripts>build script</a> 
 * provided for this example in the <font face="Courier New" size = -1>samples/examples/jdbc/oracle</font> 
 * directory. The script will perform the following step:
 * <p> 
 * <ol type=a>
 * <li>Compile the spcursors class as shown in this example for <b>Windows NT/2000</b>:
 * <p>
 * <pre>  $ <b>javac -d %CLIENT_CLASSES% spcursors.java</b></pre>
 * </ol>
 * <p> 
 * <li>Execute the stored procedures in the
 * file <font face="Courier New" size=-1><a href="spcursors.sql">spcursors.sql</a></font> in the same
 * directory as this example using Oracle SQL/Plus:
 * <p>
 * <pre>  $ <b>sqlplus scott/tiger@DEMO @spcursors.sql</b></pre>
 * </ol>
 * <p><h3>Run the Example</h3>
 * <ul>
 * <li>Execute the following command in your development shell: 
 * 
 * <pre><b>$ java examples.jdbc.oracle.spcursors</b></pre>
 * 
 * </ul>
 * <h3>There's More</h3>
 *
 * For more information about the WebLogic jDriver for Oracle, see   <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/index.html"><i>Installing and Using WebLogic jDriver for Oracle</i></a>. 
 * <p>
 * @author Copyright (c) 1996-2001 by BEA Systems, Inc.  All Rights Reserved.
 */

public class spcursors {
  
  public static void main(String[] argv)
  {
    String user = "scott";
    String password = "tiger";
    String server = "DEMO";

    try {
      for (int i = 0; i < argv.length; i++) 
      {
        if (argv[i].equals("-user")) {
          i++; 
          user = (argv[i].equals("null") ? "" : argv[i]);
        } 
        else if (argv[i].equals("-password")) {
          i++; 
          password = (argv[i].equals("null") ? "" : argv[i]);
        }
        else if (argv[i].equals("-server")) {
          i++; 
          server = (argv[i].equals("null") ? "" : argv[i]);
        }
      }
    } catch(ArrayIndexOutOfBoundsException aiobe) {
      System.err.println("\nUsage: java examples.jdbc.oracle.spcursors [options] \n\n" +
                         "where options include:\n" +
                         "    -user <user>            User name to be passed to database.\n" +
                         "    -password <password>    User password to be passed to database.\n" +
                         "    -server <server>        DNS name of database server.\n"); 
      System.exit(1);
    }

    java.sql.Connection conn  = null;
    java.sql.ResultSet rs     = null;
    java.sql.ResultSet rs1     = null;
    java.sql.ResultSet rs2     = null;
    java.sql.ResultSet rs3     = null;
    weblogic.jdbc.common.OracleCallableStatement cstmt = null;
    weblogic.jdbc.common.OracleCallableStatement cstmt2 = null;
    weblogic.jdbc.common.OracleCallableStatement cstmt3 = null;
    
    try {
      Properties props = new Properties();
      props.put("user",     user);
      props.put("password", password);
      props.put("server",   server);
      
      Driver myDriver = (Driver)
          Class.forName("weblogic.jdbc.oci.Driver").newInstance();
      conn = myDriver.connect("jdbc:weblogic:oracle", props);
    
      // Here we prepare a CallableStatement using a WebLogic extension
      // to JDBC that supports binding an Oracle cursor to an output
      // parameter.  Register the output parameter type as OTHER . . . 
      cstmt = 
        (weblogic.jdbc.common.OracleCallableStatement)conn.prepareCall("BEGIN OPEN ? FOR select * from emp; END;");
      cstmt.registerOutParameter(1, java.sql.Types.OTHER);
    
      // . . . and execute it. Retrieve the results into a ResultSet,
      // print it and close it. Then close the CallableStatement.  Note
      // that the getResultSet(int) method is a WebLogic extension for
      // retrieving the results of a stored procedure.
      cstmt.execute();
      rs = cstmt.getResultSet(1);
      printResultSet(rs);
      rs.close();
      cstmt.close();
    
      cstmt2 = 
        (weblogic.jdbc.common.OracleCallableStatement)conn.prepareCall("BEGIN single_cursor(?, ?); END;");
      cstmt2.registerOutParameter(1, java.sql.Types.OTHER);
    
      // You can also set input parameters, execute the statement,
      // retrieve the query into a ResultSet, and print it
      // for each set of input parameters.
      cstmt2.setInt(2, 1);  
      cstmt2.execute();
      rs = cstmt2.getResultSet(1);
      printResultSet(rs);
    
      cstmt2.setInt(2, 2);  
      cstmt2.execute();
      rs = cstmt2.getResultSet(1);
      printResultSet(rs);
    
      cstmt2.setInt(2, 3);  
      cstmt2.execute();
      rs = cstmt2.getResultSet(1);
      printResultSet(rs);
    
      // Close the CallableStatement
      cstmt2.close();
    
      // Here we prepare the CallableStatement,
      // register its output parameters (note the use of type OTHER
      // to bind the parameter to an Oracle cursor) . . . and execute it.
      cstmt3 = 
        (weblogic.jdbc.common.OracleCallableStatement)conn.prepareCall("BEGIN multi_cursor(?, ?, ?); END;");
      cstmt3.registerOutParameter(1, java.sql.Types.OTHER);
      cstmt3.registerOutParameter(2, java.sql.Types.OTHER);
      cstmt3.registerOutParameter(3, java.sql.Types.OTHER);
    
      cstmt3.execute();
    
      // Create a ResultSet for each set of parameters
      rs1 = cstmt3.getResultSet(1);
      rs2 = cstmt3.getResultSet(2);
      rs3 = cstmt3.getResultSet(3);
    
      // Print the ResultSets
      printResultSet(rs1);
      printResultSet(rs2);
      printResultSet(rs3);
    
      // Always close objects like Statements and ResultSets
      // when you've finished using them.
      rs1.close();
      rs2.close();
      rs3.close();
    
      // Close the CallableStatement
      cstmt3.close();

    } catch (Exception e) {
        System.out.println("Exception was thrown: " + e.getMessage());
    } finally { //close Connection and other objects in a finally block
        try {
          if(rs != null)
            rs.close();
          if(rs1 != null)
            rs1.close();
          if(rs2 != null)
            rs2.close();
          if(rs3 != null)
            rs3.close();
          if(cstmt != null)
            cstmt.close();
          if(cstmt2 != null)
            cstmt2.close();
          if(cstmt3 != null)
            cstmt.close();
          if(conn != null)
            conn.close();
        } catch (SQLException sqle) {
            System.out.println("SQLException was thrown: " + sqle.getMessage());
        }
    }
  }
  
  static void printResultSet(ResultSet rs) 
       throws SQLException
  {
    int rowcount = 0;
    while(rs.next()) rowcount++;
    System.out.println("Records = " + rowcount  );
  }
}

