package examples.jdbc.oracle;

import java.sql.*;
import java.util.Properties;

/**
 * This example demonstrates syntax and usage for Oracle
 * stored procedures and functions. The code shows how
 * to create and call stored procedures and functions
 * and how to register parameters.
 *
  * <p><h3>Build the Example</h3>  
 * <ol>
 * <li> Open a new command shell. 
 * <p><li>Set up this development shell as described in 
 * <a href=../../examples.html#environment>Setting up Your Environment for 
 * Building and Running the Examples</a>.
 * <p>
 * <li> Change connection parameters to correspond to your Oracle configuration.
 * If you need more help, check the section on connecting
 * to a database in the programming guide, <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/API_joci.html">Using WebLogic jDriver for Oracle</a>.
 * <p>
 * <ul><li>There are two pathnames required in your environment
 * 1) the directory in which the WebLogic shared library (native interface) resides, and
 * 2) the directory in which the vendor-supplied libraries from Oracle
 * reside (<i>bin</i> for NT or <i>lib</i> for UNIX). 
 * Be sure to include these pathnames in your path.
 * <p>
 * <li> For complete information, see <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#environment">Setting Up the Environment for Using 
 * WebLogic jDriver for Oracle</a>
 * in <i>Installing and Using WebLogic jDriver for Oracle</i></a>. 
 * The following sections contain instructions for specific platforms:
 * <p>
 * <ul>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#nt">Windows NT</a>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#solaris">Solaris</a>
 * <li><a
 * href="http://e-docs.bea.com/wls/docs61/oracle/install_jdbc.html#hp">HP</a>
 * </ul>
 * <p></ul>
 * <li>Compile the example by executing the following command or by executing the <a href=../../examples.html#buildScripts>build script</a> 
 * provided for this example in the <font face="Courier New" size = -1>samples/examples/jdbc/oracle</font> 
 * directory. The script will perform the following step:
 * <p> 
 * <ul>
 * <li>Compile the storedprocs class as shown in this example for <b>Windows NT/2000</b>:
 * <p>
 * <pre>  $ <b>javac -d %CLIENT_CLASSES% storedprocs.java</b></pre>
 * </ol>
 * <p><h3>Run the Example</h3>
 * <ul>
 * <li>Execute the following command in your development shell: 
 * 
 * <pre><b>$ java examples.jdbc.oracle.storedprocs</b></pre>
 * 
 * </ul>
 * <h3>There's More</h3>
 *
 * For more information about the WebLogic jDriver for Oracle, see <a
 * href="http://e-docs.bea.com/wls/docs61/oracle/index.html"><i>Installing and Using WebLogic jDriver for Oracle</i></a>. 
 * <p>
 * @author Copyright (c) 1996-2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class storedprocs {
    
    public static void main(String argv[])
    {
      String user = "scott";
      String password = "tiger";
      String server = "DEMO";

      try {
        for (int i = 0; i < argv.length; i++) 
        {
          if (argv[i].equals("-user")) {
            i++; 
            user = (argv[i].equals("null") ? "" : argv[i]);
          } 
          else if (argv[i].equals("-password")) {
            i++; 
            password = (argv[i].equals("null") ? "" : argv[i]);
          }
          else if (argv[i].equals("-server")) {
            i++; 
            server = (argv[i].equals("null") ? "" : argv[i]);
          }
        }
      } catch(ArrayIndexOutOfBoundsException aiobe) {
        System.err.println("\nUsage: java examples.jdbc.oracle.storedprocs [options] \n\n" +
                           "where options include:\n" +
                           "    -user <user>            User name to be passed to database.\n" +
                           "    -password <password>    User password to be passed to database.\n" +
                           "    -server <server>        DNS name of database server.\n"); 
        System.exit(1);
      }

      java.sql.Connection conn          = null;
      java.sql.Statement stmt1          = null;
      java.sql.Statement stmt2          = null;
      java.sql.CallableStatement cstmt1 = null;
      java.sql.CallableStatement cstmt2 = null;
      try {
        Properties props = new Properties();
        props.put("user",     user);
        props.put("password", password);
        props.put("server",   server);
        
        Driver myDriver = (Driver)   Class.forName("weblogic.jdbc.oci.Driver").newInstance();
        conn = myDriver.connect("jdbc:weblogic:oracle", props);
        
        // Create a stored proc
        stmt1 = conn.createStatement();
        stmt1.execute("CREATE OR REPLACE PROCEDURE proc_squareInt " +
                      "(field1 IN OUT INTEGER, " +
                      " field2 OUT INTEGER) IS " + 
                      "BEGIN field2 := field1 * field1; " +
                      "field1 := field1 * field1; END proc_squareInt;");
        stmt1.close();
        
        // Native Oracle SQL is commented out here
        // String sql = "BEGIN proc_squareInt(?, ?); END;";
        
        // This is correct syntax as specified in the JDBC spec
        String sql = "{call proc_squareInt(?, ?)}";
        cstmt1 = conn.prepareCall(sql);
        
        // Register out parameters
        cstmt1.registerOutParameter(2, java.sql.Types.INTEGER);
        for (int i = 0; i < 5; i++) {
            cstmt1.setInt(1, i);
            cstmt1.execute();
            System.out.println(i + " " + cstmt1.getInt(1) + " " + cstmt1.getInt(2));
        }
        cstmt1.close();
        
        // Create a function
        stmt2 = conn.createStatement();
        stmt2.execute("CREATE OR REPLACE FUNCTION func_squareInt (field1 IN INTEGER) " +
                      "RETURN INTEGER IS " +
                      "BEGIN return field1 * field1; END func_squareInt;");
        stmt2.close();
        
        // Native Oracle SQL is commented out here
        // sql = "BEGIN ? := func_squareInt(?); END;";
        
        // This is correct syntax as specified in the JDBC spec
        sql = "{ ? = call func_squareInt(?)}";
        cstmt2 = conn.prepareCall(sql);
        
        // Register out parameters
        cstmt2.registerOutParameter(1, java.sql.Types.INTEGER);
        for (int i = 0; i < 5; i++) {
            cstmt2.setInt(2, i);
            cstmt2.execute();
            System.out.println(i + " " + cstmt2.getInt(1) + " " + cstmt2.getInt(2));
        }
        cstmt2.close();
      } catch (Exception e) {
          System.out.println("Exception was thrown: " + e.getMessage());
      } finally {
          try {
            if(stmt1 != null)
              stmt1.close();
            if(stmt2 != null)
              stmt2.close();
            if(cstmt1 != null)
              cstmt1.close();
            if(cstmt2 != null)
              cstmt2.close();
            if(conn != null)
              conn.close();
          } catch (SQLException sqle) {
              System.out.println("SQLException was thrown: " + sqle.getMessage());
          }
      }
    }
}

