package examples.jndi;

import javax.naming.*;
import java.util.*;
import java.io.*;
import weblogic.common.*;
import weblogic.jndi.*;

/**
 * This example shows how to create a context with default (guest access) to the naming 
 * system, and how to create a context that has authenticated access to the 
 * naming system.
 * <p>The following sections cover what to do:
 * <p>
 * <ol>
 * <li><a href="#Build the Example">Build the Example</a>
 * <li><a href="#Run the Example">Run the Example</a>
 * </ol>
 * <h3><a name="Build the Example">Build the Example</a></h3>
 * <p>
 * <ol>
 * <li>Set up your development shell, as described in 
 * <a href=../examples.html#environment>Setting up 
 * your environment</a>.
 * <p>
 * <li> Compile the example by executing an <tt>ant</tt> build script or a command:
 * <ol>
 * <li>An <tt>ant</tt> build script is available in the <font face="Courier New" size = -1>samples/examples/jndi</font> 
 * directory. Enter the following command to execute the build script:
 * <p>
 * <pre><font face="Courier New"><b>ant</b></font></pre>
 * <li>Enter the following command to compile the InitialContextExample example:
 * <p>
 * <pre><font face="Courier New"> <b>$ javac -d %CLIENT_CLASSES% InitialContextExample.java</b></font></pre>
 * </ol>
 * </ol>
 * <h3><a name="Run the Example"></a>Run the Example</h3>
 * <ol>
 * <li>
 * <a href=../examples.html#startServer>Start the WebLogic Server</a> in a new command shell.
 * <p><li>
 * In your development shell, run the client with the following command:
 * <pre><font face="Courier New"><b>$ java examples.jndi.InitialContextExample t3://<i>hostname</i>:<i>port</i></b></font></pre>
 * where:
 * <dl>
 * <dt><i><font face="Courier New" size = -1><b>hostname</b></font></i>
 * <dd>Host name of the WebLogic Server.
 * <dt><i><font face="Courier New" size = -1><b>port</b></font></i>
 * <dd>Port where the WebLogic Server is listening for connections
 * (<font face="Courier New" size=-1>weblogic.system.ListenPort</font>) 
 * </dl>
 * </ol>
 * @author Copyright (c) 2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class InitialContextExample {
  /**
   * Runs this example from the command line.
   */
  public static void main(String[] args) {
    if (args.length > 3) {
      System.out.println("Usage:  java examples.jndiInitialContextExample "
                         + "[WebLogicURL [userName password]]");
      System.out.println("Example:java examples.jndiInitialContextExample "
                         + "t3://localhost:7001 john mypasswd");
      
    }
    else {
      int idx = 0;
      // parse the url, user, and password if given
      String url      = (args.length > idx) ? args[idx++] : null;
      String user     = (args.length > idx) ? args[idx++] : null;
      String password = (args.length > idx) ? args[idx++] : "";
      Context ctx = null;
      
      try {
        Hashtable env = new Hashtable();
        env.put(Context.INITIAL_CONTEXT_FACTORY, 
          weblogic.jndi.WLInitialContextFactory.class.getName());
        if (url != null) {
          env.put(Context.PROVIDER_URL, url);
        }
        if (user == null) {
          ctx = new InitialContext(env);
          System.out.println("WebLogic context created on behalf of \"guest\"");
        } else {
          // Package user and password in T3User to ensure that password is
          // encrypted.
          env.put(Context.SECURITY_CREDENTIALS, new T3User(user, password));
          // The following 2 lines show an alternative method for specifying name 
          // and password that transmits the password in clear text:
          //   env.put(Context.SECURITY_PRINCIPAL, user);
          //   env.put(Context.SECURITY_CREDENTIALS, password);
          ctx = new InitialContext(env);
          System.out.println("WebLogic context created on behalf of \"" + user + "\"");
        }
      }
      catch (AuthenticationException e) {
        System.out.println("You've specified an invalid user name or password");
      }
      catch (CommunicationException e) {
        System.out.println(
          "Failed to contact " + 
          ((url == null) ? "t3://localhost:7001" : url) + ".\n" +
          "Is there a server running at this address?"
        );
      }
      catch (Exception e) {
        System.out.println("An unexpected exception occurred:" +e);
        e.printStackTrace();
      }
      finally {
        if (ctx != null) {
        // Always close context when finished.
          try {
            ctx.close();
          }
          catch (NamingException e) {
            System.out.println("Failed to close context due to: " + e);
          }
        }
      }
    }
  }
}  









