package examples.security.acl;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.StringTokenizer;
import java.util.Vector;
import javax.naming.Context;
import weblogic.jndi.Environment;
import weblogic.security.PEMInputStream;
import weblogic.security.X509;
import weblogic.security.acl.DefaultUserInfoImpl;


/** 
 * This simple client connects as a specific user and attempts an RMI
 * invocation. Access to the invocation will be granted or denied
 * depending on the configured ACLs. <p>
 *
 * This differs from the Client example in two ways:
 * <ul>
 *
 * <li>It uses Environment to establish the initial JNDI context
 * instead of InitialContext and a Hashtable.
 *
 * <li>It attempts to perform two-way SSL authentication, if a client
 * private key and certificates are provided.
 *
 * </ul>
 *
 * <p>If you want to test two-way SSL authentication, you can use the
 * <code>demokey.pem</code> and <code>democert.pem</code> files
 * provided in your installation as your client key and certificate.
 * Don't forget to configure the server to require two-way SSL
 * authentication; it should use the <code>ca.pem</code> file (also
 * in your installation) as the Certification Authority to check for
 * when clients connect.
 *
 * @author Copyright (c) 2000-2001 by BEA Systems, Inc. All Rights Reserved.
 * @see <a href="Client.html">Client</a>
 */
public class AltClient
{
  /** 
   * The main entry point.  The summary is that we check our
   * parameters, look up <tt>frobtarget</tt>, call <tt>frob</tt> on
   * it, and report on whether it succeeded.
   */
  public static void main(String[] args)
  {
    Context ctx = null;
    
    if (args.length < 1)
    {
      usage();
    }

    String url = args[0];
    
    try
    {
      Environment env = new Environment();

      env.setProviderUrl(url);

      // If we've been given an explicit username and password, use
      // them.  Otherwise, the JNDI connection will attempt to default
      // to guest/guest.

      String user = null;
      
      for (int arg = 1; arg < args.length; arg++)
      {
        if (args[arg].equals("-user") && arg + 1 < args.length)
        {
          env.setSecurityPrincipal(user = args[++arg]);
        }
        else if (args[arg].equals("-pass") && arg + 1 < args.length)
        {
          env.setSecurityCredentials(args[++arg]);
        }
        else if (args[arg].equals("-sslCert") && arg + 1 < args.length)
        {
          // If we've been asked to make a secure T3 connection and we
          // have enough arguments, we populate the SSL client
          // certificate so that we can attempt two-way SSL authentication.

          // In order to do this, we must make an array of at least two
          // InputStream objects; the first is the client's private key,
          // and the rest are the certificate chain, starting with the
          // client's certificate (which must be present) and ending with
          // the root CA's certificate.

          InputStream[] certs = readCerts(args[++arg]);

          if (url.startsWith("t3s") || url.startsWith("https"))
          {
            env.setSSLClientCertificate(certs);
          } else {
            fatal("the URL doesn't specify use of SSL");
          }
        }
        else if (args[arg].equals("-servername") && arg + 1 < args.length)
        {
          if (url.startsWith("t3s") || url.startsWith("https"))
          {
            // This turns on peer-validation which means that we'll check
            // that the hostname specified in the certificate sent to us
            // by the server actually matches the name we supply as an
            // arg here.  Note that if the name does not match we will get
            // an error that no server was found at the url we specified.
            env.setSSLServerName(args[++arg]);
          } else {
            fatal("the URL doesn't specify use of SSL");
          }

        }
        else if (args[arg].equals("-keypass") && arg + 1 < args.length)
        {
          if (url.startsWith("t3s") || url.startsWith("https"))
          {
            // If we are using an RSA encrypted private key for the
            // client (PKCS5/PKCS8) this provides a mechanism to unlock
            // the private key if we are doing Mutual Authentication
            // (which is turned on by setting ClientCertificateEnforced
            // to true in the server).  If we are using an unencrypted
            // RSA private key then this option is not required.
            env.setSSLClientKeyPassword(args[++arg]);
          } else {
            fatal("the URL doesn't specify use of SSL");
          }
        }
        else if (args[arg].equals("-cert") && arg + 1 < args.length)
        {
          if (user == null)
          {
            fatal("user name must be specified before certificate chain");
          }
          
          InputStream[] certs = readCerts(args[++arg]);
          X509[] x509 = new X509[certs.length];

          for (int i = 0; i < certs.length; i++)
          {
            x509[i] = new X509(certs[i]);
          }

          env.setSecurityCredentials(new DefaultUserInfoImpl(user, x509));
        } else {
          usage();
        }
      }

      ctx = env.getInitialContext();

      Frobable f = (Frobable) ctx.lookup("frobtarget");
      f.frob();
      System.out.println("Frobbed successfully");
    }
    catch (Throwable t)
    {
      t.printStackTrace();
      System.out.println("Failed to frob");
    }
    finally
    {
      try
      {
        if (ctx != null)
        {
          ctx.close();
        }
      }
      catch (Exception e)
      {
        // Deal with any failures
      }
    }
  }


  private static final String pathSep =
    (String) System.getProperty("path.separator", ":");

  
  private static InputStream[] readCerts(String files)
    throws IOException
  {
    Vector inputStreams = new Vector();

    StringTokenizer toks = new StringTokenizer(files, pathSep);

    // A pem encoded file may have multiple certs inside it
//    InputStream[] streams = new InputStream[toks.countTokens()];

    int num_files = toks.countTokens();
    for (int i = 0; i < num_files; i++)
    {
      String file = toks.nextToken();


      InputStream is = new FileInputStream(file);

      if (file.toLowerCase().endsWith(".pem")) {
        // Files that are in pem format may contain more than
        // one pem encoded cert. The minimum size for a PEM is 64
        // (see PEMInputStream)
        while ( is.available() > 64 ) {
          // While we still possibly have a cert keep sucking in
          // certs from the input stream
          PEMInputStream pis = new PEMInputStream(is);
          inputStreams.addElement(pis);
        }
      } else {
        // We don't have a pem encoded file so we just have one cert
        // (der encoded files can only contain one cert)
        inputStreams.addElement(is);
      }

    } // done with for loop

    // We've got to create an appropriate array of InputStream to
    // return to the caller so transfer info from our vector of
    // InputStreams (both regular InputStreams and possibly PEMInputStreams)
    InputStream streams[] = new InputStream[inputStreams.size()];
    for (int i = 0; i < inputStreams.size(); i++ ) {
      streams[i] = (InputStream)inputStreams.elementAt(i);
    }
    
    return streams;
  }
  
  private static void fatal(String msg)
  {
    System.err.println("Error: " + msg);
    System.exit(1);
  }

  
  private static void usage()
  {
    System.err.println("Usage:\tjava examples.security.acl.AltClient " +
        	       "URL [-user username]");
    System.err.println("\t[-pass password] [-cert cert1" + pathSep + "cert2" +
        	       pathSep + "...]");
    System.err.println("\t[-servername hostnameincert] [-sslCert key" + pathSep + "cert1" +
        	       pathSep + "... [-keypass privatekeypassword] ]");
    System.err.println("e.g.:\tjava examples.security.acl.AltClient " +
        	       "t3s://localhost:7002 -user guest -pass guest -servername myhost.com");
    System.exit(1);
  }
}
