
/**
 * ConfigParser.java
 *
 * Jaas configuration parser
 *
 * @author Copyright (c) 2000-2001 by BEA Systems, Inc. All Rights Reserved.
 */

package examples.security.jaas;

import javax.security.auth.login.AppConfigurationEntry;
import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.util.StringTokenizer;
import java.util.Hashtable;
import java.util.Enumeration;

class ConfigParser
{
  /**
   * Utility for parsing JAAS loginModule configuration policy files
   */
  public static AppConfigurationEntry[] parse(String configFileName, String applicationName)
  {
    AppConfigurationEntry[] list = null;
    AppConfigurationEntry[] tmpList = new AppConfigurationEntry[100];
    AppConfigurationEntry entry = null;
    int entryIdx = 0;
    int lineIdx = 0;

    try
    {
      FileReader fileReader = new FileReader(configFileName);
      BufferedReader bufferedReader = new BufferedReader(fileReader);
      String line = null;

      String loginModuleName = null;
      AppConfigurationEntry.LoginModuleControlFlag controlFlag = null;
      Hashtable options = new Hashtable();

      // Ignore everything until we find the specified applicationName
      boolean appFound = false;
      while((line = bufferedReader.readLine()) != null)
      {
        lineIdx++;

        StringTokenizer st = new StringTokenizer(line, " ;\t\n\r\f", true);
        while(st.hasMoreTokens())
        {
          String value = st.nextToken();

          // Look for application entry name
          if(value.equals(applicationName))
          {
            appFound = true;
            continue;
          }

          // Starting config block delimiter
          if(appFound && value.equals("{"))
          {
            continue;
          }

          // Terminating config block delimiter
          if(appFound && value.equals("}"))
          {
            bufferedReader.close();

            list = new AppConfigurationEntry[entryIdx];
            for(int i = 0; i < entryIdx; i++)
              list[i] = tmpList[i];

            return list;
          }

          // White space
          if(value.equals(" "))
             continue;

          // Parse individual config entry elements
          if(appFound)
          {
            if(value.equals(";"))
            {
              if((loginModuleName == null) || (controlFlag == null))
                System.out.println("Warning: bad configuration definition, line " + lineIdx);
              else
              {
                entry = new AppConfigurationEntry(loginModuleName, controlFlag, options);
                tmpList[entryIdx++] = entry;
              }

              // Reset values for next module configuration entry
              loginModuleName = null;
              controlFlag = null;
              options = new Hashtable();
              continue;
            }

            // Set loginModuleName
            if(loginModuleName == null)
            {
              loginModuleName = value;
              continue;
            }

            // Set controlFlag
            if(controlFlag == null)
            {
              if(value.equalsIgnoreCase("OPTIONAL"))
              {
                controlFlag = AppConfigurationEntry.LoginModuleControlFlag.OPTIONAL;
              }
              else if(value.equalsIgnoreCase("REQUIRED"))
              {
                controlFlag = AppConfigurationEntry.LoginModuleControlFlag.REQUIRED;
              }
              else if(value.equalsIgnoreCase("REQUISITE"))
              {
                controlFlag = AppConfigurationEntry.LoginModuleControlFlag.REQUISITE;
              }
              else if(value.equalsIgnoreCase("SUFFICIENT"))
              {
                controlFlag = AppConfigurationEntry.LoginModuleControlFlag.SUFFICIENT;
              }
              else
              {
                throw new IllegalArgumentException("Invalid controlFlag");
              }
              continue;
            }

            // Set options
            String key = null;
            String val = null;

            //System.out.println(value);

            StringTokenizer stOpt = new StringTokenizer(value, " =\t\n\r\f");
            while(stOpt.hasMoreTokens())
            {
              String value2 = stOpt.nextToken();
              if(key == null)
              {
                key = value2;
                continue;
              }
              if(val == null)
              {
                val = value2;
                continue;
              }
            }
            if((key != null) && (val != null))
            {
              options.put(key, val);
            }
          }
        }
      }
      bufferedReader.close();
    }
    catch(java.io.FileNotFoundException ioe)
    {
      System.out.println(ioe.toString());
    }
    catch(java.io.IOException ioe)
    {
      System.out.println(ioe.toString());
    }
    return list;
  }

  /*
   * Unit test main, needs a test.policy file to parse
   */
  public static void main(String argv[])
  {
    String policyFile = "test.policy";
    AppConfigurationEntry[] policy = null;

    policy = ConfigParser.parse(policyFile, "SamplePolicy");
    System.out.println("SamplePolicy");
    for(int i = 0; i < policy.length; i++)
    {
      System.out.print("  " + policy[i].getLoginModuleName() + " ");
      System.out.print(policy[i].getControlFlag().toString());
      Hashtable table = (Hashtable) policy[i].getOptions();
      Enumeration enum = table.keys();
      while(enum.hasMoreElements())
      {
        String key = (String) enum.nextElement();
        String value = (String) table.get(key);
        System.out.print(" " + key + "=" + value);      
      }
      System.out.println(" ");
    }
    System.out.println(" ");

    policy = ConfigParser.parse(policyFile, "CertSamplePolicy");
    System.out.println("CertSamplePolicy");
    for(int i = 0; i < policy.length; i++)
    {
      System.out.print("  " + policy[i].getLoginModuleName() + " ");
      System.out.print(policy[i].getControlFlag().toString());
      Hashtable table = (Hashtable) policy[i].getOptions();
      Enumeration enum = table.keys();
      while(enum.hasMoreElements())
      {
        String key = (String) enum.nextElement();
        String value = (String) table.get(key);
        System.out.print(" " + key + "=" + value);      
      }
      System.out.println(" ");
    }
  }
}

