package examples.security.jaas;

import java.security.acl.Acl;
import java.security.acl.Permission;
import java.util.Hashtable;
import javax.naming.Context;
import javax.naming.InitialContext;
import java.rmi.RemoteException;
import weblogic.security.acl.BasicRealm;
import weblogic.security.acl.Realm;
import weblogic.security.acl.Security;
import weblogic.security.acl.User;
import weblogic.security.audit.Audit;

/**
 * Illustrates how to use an ACL to
 * protect the methods of an RMI object's. You will see either a success
 * message or a SecurityException from the Frob method.
 *
 * @author Copyright (c) 1997-2001 by BEA Systems, Inc. All Rights Reserved.
 */
public class FrobImpl implements Frobable {

  final static String ACL_NAME = "aclexample";
  final static String PERMISSION_NAME = "frob";
  final static String REALM_NAME = "weblogic";

  /**
   * Illustrates two ways to check the permissions of an object.
   * Checking permission is a single static method call for ACLs in
   * a security realm. You can also use a general
   * method of checking security for the current user in the
   * current security realm.
   */
  public void frob() throws RemoteException {

    // For ACLs in the weblogic realm, you can use static methods in Security.
    Security.checkPermission("Frob",
			     Security.getCurrentUser(), 
			     ACL_NAME, 
			     Security.getRealm().getPermission(PERMISSION_NAME),
			     '.');

    // In general, the following works, but you're better off using
    // the static methods because they will audit for you.
    User p = Security.getCurrentUser();
    BasicRealm realm = Realm.getRealm(REALM_NAME);
    Acl acl = realm.getAcl(ACL_NAME);
    Permission perm = realm.getPermission(PERMISSION_NAME);
    boolean result = acl == null || !acl.checkPermission(p, perm);
    
    Audit.checkPermission("Frob", acl, p, perm, !result);
    
    if (result) {
      Security.logAndThrow(PERMISSION_NAME + " denied for user " + p);
    }
    System.out.println("User " + p + " frobbed successfully");
  }    
  
  /**
   * Creates a FrobImpl and binds in JNDI as <tt>frob target</tt>.
   */
  public static void main(String args[]) {
    Context ctx = null;
    try {
      Hashtable env = new Hashtable();
      env.put(Context.INITIAL_CONTEXT_FACTORY,
              "weblogic.jndi.WLInitialContextFactory");
      ctx = new InitialContext(env);
      ctx.bind("frobtarget", new FrobImpl());
    }
    catch (Throwable t) {
      t.printStackTrace();
    }
    finally {
      try {ctx.close();}
      catch (Exception e) {
        // Deal with any failure
      }
    }
  }
}
