package examples.security.rdbmsrealm;


import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import weblogic.security.acl.BasicRealm;
import weblogic.security.acl.User;
import weblogic.utils.encoders.BASE64Encoder;


/**
 * This is the in-memory representation of a user from the database.
 * The user has a name and password.  The password may be one-way
 * hashed. Hashed passwords are stored in the database in the form
 * <tt>{HASH}gobbledygook</tt>, where <tt>HASH</tt> is the name of the
 * hashing algorithm, and <tt>gobbledygook</tt> is the hashed
 * password. <p>
 *
 * You can store passwords either in hashed or plaintext form.
 * You can vary how passwords are stored on a per-user basis (for example, use
 * MD5 for one user and SHA for another user). 
 * However, BEA recommends using a single hashing algorithm for all passwords and
 * not storing any plain text passwords in the database. <p>
 *
 * This class has a <tt>main</tt> method, which you can use to
 * generate hashed passwords from plain text passwords.  The hashed
 * passwords can then be stored in your database.
 *
 * @author Copyright (c) 1998-2001 by BEA Systems, Inc. All Rights Reserved.
 */
class RDBMSUser
  extends User
{
  /**
   * The default password generation algorithm.
   */
  protected static final String ALGORITHM = "SHA";

  /**
   * The realm that created this user object.
   */
  private transient RDBMSRealm realm;

  /**
   * The user's password.  If the password is hashed, the md field will
   * contain an instance of an object that performs the hashing
   * algorithm.
   */
  private transient String passwd;

  /**
   * The digest algorithm used to one-way hash the user's password.
   * If the password is not hashed with a known algorithm, or is in
   * plain text, this will be null.
   */
  private transient MessageDigest md;


  /**
   * Creates a user with the given name and hashed password
   * from the given realm.
   */
  RDBMSUser(String name, String passwd, RDBMSRealm realm)
  {
    super(name);
    this.realm = realm;

    // Checks to see if the password was stored in hashed form, and if
    // it was, whether we can use that hashing algorithm.

    if (passwd != null)
    {
      int rightCurly = passwd.indexOf("}");
    
      if (rightCurly > 0 && passwd.charAt(0) == '{')
      {
	this.passwd = passwd.substring(rightCurly + 1);

	String algorithm = passwd.substring(1, rightCurly);

	try
	{
	  md = MessageDigest.getInstance(algorithm.toUpperCase());
	}
	catch (NoSuchAlgorithmException e)
	{
	  if (realm.log != null)
	  {
	    realm.log.error("digest algorithm \"" + algorithm +
			    "\" not found - assuming plaintext password");
	  } else {
	    System.err.println("Error: digest algorithm \"" + algorithm +
			       "\" not found - assuming plaintext password");
	  }
	}
      } else {
	this.passwd = passwd;
	this.md = null;
      }
    }
  }


  /**
   * Returns the realm that created this object.
   */
  public BasicRealm getRealm()
  {
    return realm;
  }

  
  /**
   * Hashes the given plain text with the given digest algorithm, and
   * base64-encode the result.
   *
   * @param md message digest algorithm to hash with
   * @param plaintext text to hash
   * @return base64-encoded hashed text
   */
  static protected String hash(MessageDigest md, String plaintext)
  {
    BASE64Encoder enc = new BASE64Encoder();
      
    return enc.encodeBuffer(md.digest(plaintext.getBytes()));
  }

  
  /**
   * Checks a plain text password against the user's password.  If the
   * object containing the password is not known, authentication will
   * fail.
   *
   * @param plaintext the plaintext password to check
   * @return true if matched, false otherwise
   */
  boolean authenticate(String plaintext)
  {
    String hashed = md != null ? hash(md, plaintext) : plaintext;

    return hashed.equals(passwd);
  }
  

  /**
   * Hashes passwords according to the given algorithm.  Plain text
   * passwords are read from stdin, and the encrypted passwords are
   * printed to stdout.  If no algorithm is specified on the command
   * line, the one specified in ALGORITHM is used.
   *
   * @see #ALGORITHM
   */
  public static void main(String[] args)
    throws IOException
  {
    String algorithm = (args.length >= 1 ? args[0] : ALGORITHM).toUpperCase();
    BufferedReader r = new BufferedReader(new InputStreamReader(System.in));
    MessageDigest md = null;
    String prefix = null;
    String plaintext;

    try
    {
      md = MessageDigest.getInstance(algorithm);
    }
    catch (NoSuchAlgorithmException e)
    {
      // ignore
    }

    if (md == null)
    {
      System.err.println("Error: unknown algorithm \"" + algorithm + "\"");
      System.exit(1);
    }
      
    System.err.println("Enter plaintext passwords, separated by newlines.");
    
    while ((plaintext = r.readLine()) != null)
    {
      String passwd = "{" + algorithm + "}" + hash(md, plaintext);
      
      System.out.println(passwd);

      if (System.out.checkError())
      {
	throw new IOException("output error");
      }
    }

    r.close();
  }
}
