(* core.sml *)

structure Core =
struct

local 
open Syntax
open Error
in

(* ------------------------   EVALUATION  ------------------------ *)

exception NoRuleApplies

(* values are terms of the form TmTrue, TmFalse, or TmAbs _
   Note: variables are not considered values in this version. *)
fun isval t =
    case t
      of (TmTrue | TmFalse | TmAbs _) => true
       | _ => false

fun eval1 ctx t =
    case t
      of TmIf(TmTrue,t2,t3) => t2
       | TmIf(TmFalse,t2,t3) => t3
       | TmIf(t1,t2,t3) => TmIf(eval1 ctx t1, t2, t3)
       | TmApp(v1 as TmAbs(_,_,t12), t2) =>
           if isval t2 then termSubstTop(t2,t12)
	   else TmApp(v1, eval1 ctx t2)
       | TmApp(t1,t2) =>
	   TmApp(eval1 ctx t1, t2)
       | _ => raise NoRuleApplies

fun eval ctx t =
    let val t' = eval1 ctx t
     in eval ctx t'
    end
    handle NoRuleApplies => t


(* ------------------------   TYPING  ------------------------ *)

fun typeof(ctx,t) =
    case t
      of TmTrue => TyBool
       | TmFalse => TyBool
       | TmIf(t1,t2,t3) =>
           if equalTy (typeof(ctx,t1), TyBool)
           then let val tyT2 = typeof(ctx,t2)
	         in if equalTy(tyT2,typeof(ctx,t3)) then tyT2
                    else error "arms of conditional have different types"
		end
           else error "guard of conditional not a boolean"
  | TmVar(i,_) => getTypeFromContext(ctx,i)
  | TmAbs(x,tyT1,t2) =>
      let val ctx' = addbinding(ctx,x,VarBind(tyT1))
          val tyT2 = typeof(ctx',t2)
       in TyArr(tyT1, tyT2)
      end
  | TmApp(t1,t2) =>
      let val tyT1 = typeof(ctx,t1)
          val tyT2 = typeof(ctx,t2)
       in case tyT1
            of TyArr(tyT11,tyT12) =>
                if equalTy (tyT2, tyT11) then tyT12
                else error "parameter type mismatch"
             | _ => error "arrow type expected"
      end

end (* local *)
end (* structure Core *)
