(* structure Main: The main program. 
   Builds and connects lexers and parsers,
   implements a simple top-level evaluation loop (Main.main).
   
   For most experiments with the implementation, it should not be
   necessary to change this file.
*)

structure Main =
struct

  open PPUtil
  open Syntax
  open Core

(* Boilerplate code for setting up lexer and parser.
 * We apply the functors generated from simplebool.lex and simplebool.grm to produce
 * the SimpleBoolParser structure.
 *)

  structure SimpleBoolLrVals =
    SimpleBoolLrValsFun(structure Token = LrParser.Token)

  structure SimpleBoolLex =
    SimpleBoolLexFun(structure Tokens = SimpleBoolLrVals.Tokens)

  structure SimpleBoolParser =
    Join(structure LrParser = LrParser
	 structure ParserData = SimpleBoolLrVals.ParserData
	 structure Lex = SimpleBoolLex)

(* 
 * We need a function which given a lexer invokes the parser. The
 * function invoke does this.
 *)

  fun invoke lexstream =
      let fun print_error (s,i:int,_) =
	      TextIO.output(TextIO.stdOut,
			    "Error, line " ^ (Int.toString i) ^ ", " ^ s ^ "\n")
       in SimpleBoolParser.parse(0,lexstream,print_error,())
      end

(* 
 * Finally, we need a driver function that reads one or more expressions
 * from the standard input. The function parse, shown below, does
 * this. It runs the evaluator on the standard input and terminates when
 * an end-of-file is encountered.
 *)

  fun parse () = 
      let val lexer = SimpleBoolParser.makeLexer (fn _ => TextIO.inputLine TextIO.stdIn)
       in #1(invoke lexer)
      end

  fun process_command(cmd,ctx) =
       (case cmd
	 of Eval(t) => 
             let val tyT = typeof(ctx,t)
	         val t' = eval ctx t
	      in printtm_ATerm(true,ctx,t'); 
                 print_break 1 2;
                 pr ": ";
                 printty tyT;
		 print_newline();
		 ctx
	     end
	  | Bind(x,bind) => 
	     (pr "bind "; pr x; pr " "; prbinding(ctx,bind); print_newline();
	      addbinding(ctx,x,bind)))

  fun main () =
      let val (cmds,ctx) = parse () emptycontext
       in List.foldl process_command emptycontext cmds
      end

end (* structure Main *)
