/* ArrayList
 *
 * Implementation of a homogeneous List using an array
 *
 * Operations:
 *  length
 *  size
 *  append
 *  prepend
 *  insert_at                                         
 *  remove_at
 *  is_empty
 *  at(i)
 */

#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <errno.h>

#include "ArrayList.h"

// Create a new ArrayList
ArrayList create(void) {
    ArrayList array;
    array.array = malloc(4 * sizeof(*array.array));
    if (!array.array) {
        perror("Unable to allocate memory");
        abort();
    }
    array.length = 0;
    array.size = 4;
    
    return array;
}

// Destroy an ArrayList
void destroy(ArrayList array) {
    free(array.array);
}

// Takes in an ArrayList and returns the number of elements
size_t length(const ArrayList array) {
    return array.length;
}

// Takes in an ArrayList and returns its maximum size
size_t size(const ArrayList array) {
    return array.size;
}

// Inserts elem at array[i] = v
// insert_at(&array, v, i)
void insert_at(ArrayList *array, const int elem, const size_t i) {
    // If position is greater than length, show error
    if (i > array->length) {
        fprintf(stderr, "Attempting to insert at position %lu of an ArrayList of length %lu", i, array->length);
        abort();
    }
    
    // Do I need to reallocate
    if (array->length == array->size) {
        array->array = realloc(array->array, 2 * array->size * sizeof(*array->array));
        if (!array->array) {
            perror("Unable to allocate memory");
            abort();
        }
        array->size = 2 * array->size;
    }

    // Copy over all following elements
    for (size_t j = array->length; j > i; j--) {
        array->array[j] = array->array[j - 1];
    }

    // Assign element
    array->array[i] = elem;
    array->length++;
}

void append(ArrayList *array, const int elem) {
    insert_at(array, elem, array->length);
}

void prepend(ArrayList *array, const int elem) {
    insert_at(array, elem, 0);
}

int at(const ArrayList array, const size_t i) {
    // Check if valid position
    if (i >= array.length) {
        fprintf(stderr, "Index %lu is out of bounds for ArrayList of length %lu\n", i, array.length);
        abort();
    }
    return array.array[i];
}

int remove_at(ArrayList *array, size_t i) {
    int element = at(*array, i);
    for (size_t j = i; j < array->length - 1; j++) {
        array->array[j] = array->array[j + 1];
    }
    array->length--;

    return element;
}

int is_empty(const ArrayList array) {
    return array.length == 0;
}

void print_list(const ArrayList array) {
    printf("[");
    for (size_t i = 0; i < array.length; i++) {
        printf("%d", array.array[i]);
        if (i < array.length - 1) {
            printf(", ");
        }
    }
    printf("]\n");
}
